<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Components\FilesManager\FilesManager as FilesManagerComponent;
use Core\Model\FilesManager\FilesManager;
use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Traits\TInputExtend;
use Core\Model\UI\Form\Traits\TTextBase;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Utils\Html;

class FilesManagerInput extends \Nette\Forms\Controls\TextInput
{
	use TInputExtend;
	use TTextBase;

	/** @var FilesManager */
	public $filesManager;

	/** @var Container */
	private $container;

	/** @var FilesManagerComponent */
	private $controller;

	/** @var string */
	private $controllerName;

	public function __construct($caption = null, FilesManager $filesManager, Session $session, Request $request)
	{
		parent::__construct($caption);
		$this->filesManager = $filesManager;
		$this->controller   = new FilesManagerComponent($this->filesManager, $session, $request);

		$this->monitor(BaseForm::class, function() {
			$this->attached($this->getForm());
		});
	}

	protected function attachController(): void
	{
		$name = 'filesManagerController';
		foreach (explode('-', Strings::webalize($this->getHtmlName())) as $v)
			$name .= ucfirst($v);
		$this->controllerName = $name;
		if (!$this->form->getComponent($this->controllerName, false)) {
			$this->form->addComponent($this->controller, $this->controllerName);
		}
	}

	protected function attached($form): void
	{
		parent::attached($form);
		$this->attachController();
		$currGroup = $this->getForm()->getCurrentGroup();

		$this->getForm()->setCurrentGroup(null);
		$this->getForm()->setCurrentGroup($currGroup);
	}

	public function addSecuredRoot($roots): self
	{
		$this->controller->addSecuredRoot($roots);
		return $this;
	}

	public function getControl($content = null): Html
	{
		$this->attachController();
		$this->controller->setModal();
		$langs       = $this->getForm()->langsService->getLangs(false);
		$isMultilang = $this->getIsMultiLanguage();
		$mainWrap    = Html::el();

		foreach ($isMultilang ? $langs : ['' => null] as $k => $lang) {
			$input   = parent::getControl();
			$inputId = $this->getHtmlId() . ($k ? '-' . $k : '');

			$wrapper = Html::el('div', [
				'class'         => 'filemanager-control',
				'data-base-id'  => $this->getHtmlId(),
				'data-input-id' => $inputId,
			]);
			if ($k)
				$wrapper->attrs['data-content-lang'] = $k;

			$input->attrs['type'] = 'hidden';
			if ($k)
				$input->attrs['name'] .= "[$k]";
			$input->attrs['id'] = $inputId;

			$btnThumbnail = Html::el('a', [
				'href'   => 'javascript:;',
				'target' => '_blank',
				'class'  => 'btn btn-info filesManagerInput-thumbnail',
				'id'     => $inputId . '-thumbnail',
			])->addHtml('<i class="fa fa-file"></i>');

			$fakeInput = Html::el('input', [
				'type'     => 'text',
				'readonly' => 'true',
				'class'    => 'form-control filesManagerInput-fakeInput',
				'id'       => $inputId . '-fake-input',
			]);

			$btns = Html::el('div', ['class' => 'btn-group']);

			$btn = Html::el('a', [
				'href'  => $this->controller->link('showModal!', [$this->controllerName]),
				'class' => 'ajax btn btn-success filesManagerInput-showModal',
				'id'    => $inputId . '-show-modal',
			])->setHtml($content === null ? $this->translate('default.select') : $content);

			$btnDelete = Html::el('a', [
				'href'  => '#',
				'class' => 'btn btn-danger btn-remove-image-' . $inputId,
			])->addHtml('<i class="fa fa-times"></i>');

			$btns->addHtml($btnThumbnail);
			$btns->addHtml($fakeInput);
			$btns->addHtml($btn);
			$btns->addHtml($btnDelete);

			ob_start();
			$this->controller->render();
			$htmlContent = ob_get_clean();

			$latte = file_get_contents(__DIR__ . '/FileManagerInput.latte');
			$latte = str_replace('{$inputId}', $inputId, $latte);
			$latte = str_replace('{$baseId}', $this->getHtmlId(), $latte);
			$latte = str_replace('{$controllerName}', $this->controllerName, $latte);

			$wrapper->addHtml($btns);
			$wrapper->addHtml($htmlContent);
			$wrapper->addHtml($latte);
			$wrapper->addHtml($input);

			if ($this->getValue()) {
				$value = $this->getValue();
				if (is_array($value) && $this->getIsMultiLanguage())
					$value = $value[$k] ?? null;

				$btnThumbnail->attrs['href'] = $value;
				$fakeInput->attrs['value']   = $value;
				$input->attrs['value']       = $value;
			} else {
				$btnDelete->style = ' display: none;';
			}
			$mainWrap->addHtml($wrapper);
		}

		return $mainWrap;
	}

	public function setContainer(Container $container)
	{
		$this->container = $container;
	}

	public static function register(\Nette\DI\Container $systemContainer)
	{
		$class = __CLASS__;
		\Nette\Forms\Container::extensionMethod("addFilesManager", function(
			\Nette\Forms\Container $container, $name, $caption
		) use ($class, $systemContainer) {
			/** @var FilesManagerInput $comp */
			$comp = new $class($caption,
				$systemContainer->getByType(FilesManager::class, false),
				$systemContainer->getByType(Session::class, false),
				$systemContainer->getByType(Request::class, false));

			$comp->container = $systemContainer;

			$container->addComponent($comp, $name);

			return $comp;
		});
	}
}
