<?php declare(strict_types = 1);

namespace Core\DI;

use Core\FrontModule\Model\Providers\ISearchItem;
use Core\Model\Images\Macros\Macros;
use Core\Model\Router\Route;
use Core\Model\StaticTexts;
use Core\Model\Translation\Loaders\NeonDoctrineMixedLoader;
use Kdyby;
use Nette;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Nette\PhpGenerator\ClassType;

class CoreExtension extends CompilerExtension
{
	const TAG_ROUTER = 'app.router';

	public function loadConfiguration()
	{
		$this->setConfig($this->loadFromFile(__DIR__ . '/config.neon'));

		$builder = $this->getContainerBuilder();
		$builder->addDefinition('application.presenterFactoryCallback')
			->setFactory(
				'Nette\Bridges\ApplicationDI\PresenterFactoryCallback',
				[
					'@container',
					Nette\Application\UI\Presenter::INVALID_LINK_WARNING,
					false,
				]
			);



		$builder->removeDefinition('application.presenterFactory');
		$builder->addDefinition('application.presenterFactory')
			->setFactory(
				'\Core\Model\Application\PresenterFactory',
				['@application.presenterFactoryCallback']
			);

		$builder->addDefinition('translation.neonLoader')
				->setFactory(NeonDoctrineMixedLoader::class);
	}

	public function beforeCompile()
	{
		parent::beforeCompile();

		$this->addRouters();
		$this->setMapping(['Core' => 'Core\*Module\Presenters\*Presenter']);

		if (isset($this->compiler->getConfig()['application']['mapping']))
			$this->setMapping($this->compiler->getConfig()['application']['mapping']);

		$builder = $this->getContainerBuilder();

		$builder->getDefinition('nette.latteFactory')
			->addSetup(Macros::class . '::install(?->getCompiler(), ?)', ['@self',
				$builder->parameters['system']['images']]);

		$builder->getDefinition('nette.latteFactory')
			->addSetup('addProvider', ['templateText', $builder->getDefinition('provider.templateText')]);

		$builder->getDefinition('templateTextTypesCollection')
			->addSetup(new Nette\DI\Statement('$service->setItems(?)', [$builder->findByType(ITemplateTextType::class)]));

		$service = $builder->getDefinition('core.front.searchItemsCollection');
		$service->addSetup(new Nette\DI\Statement('$service->loadItems(?)', [$builder->findByType(ISearchItem::class)]));

		$builder->getDefinition('usersUpdateAcl')->addSetup('setAclData', [$builder->parameters['acl'] ?: []]);

		$builder->getDefinition('core.admin.components.navigation')->addSetup('setData', [$builder->parameters['adminNavigation']]);

		$builder->getDefinition('translation.translator')
				->addSetup('addLoader', ['neon', $builder->getDefinition('translation.neonLoader')]);
	}

	/**
	 * @param ClassType $class
	 */
	public function afterCompile(ClassType $class)
	{
		$init = $class->methods["initialize"];
		$builder = $this->getContainerBuilder();

		$init->addBody('\Core\Model\UI\Form\Controls\FilesManagerInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('\Core\Model\UI\Form\Controls\EditorInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('\Core\Model\UI\Form\Controls\LangsSelectInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('\Core\Model\UI\Form\Controls\GalleryPopupInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('define("CORE_TEST_MODE", $this->parameters["system"]["testMode"]);');
		$init->addBody('$this->getService(\'defaultLang\')->locale = $this->getService(\'translation.translator\')->getLocale();');

		try {
			$navigationFactory = $class->getMethod('createServiceCore__navigationFactory');
			$navigationFactory->setBody(str_replace('return $service;', '$service->setData($this->container->parameters["navigation"]);' . "\n" . 'return $service;', $navigationFactory->getBody()));
		} catch (\Exception $e) {
		}

		$init->addBody('
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$getUserEmails = function() { return $this->getService(\'users.users\')->getUserMailWithLogPrivilege(); };
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$settings = $this->getService(\'core.settings\');
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$developersMail = ?;
		', [
			$builder->parameters['system']['error']['email']
		]);
	}

	private function addRouters()
	{
		$builder = $this->getContainerBuilder();
		// Get application router
		$router = $builder->getDefinition('router');
		// Init collections
		$routerFactories = [];
		foreach ($builder->findByTag(self::TAG_ROUTER) as $serviceName => $priority) {
			// Priority is not defined...
			if (is_bool($priority)) {
				// ...use default value
				$priority = 100;
			}
			$routerFactories[$priority][$serviceName] = $serviceName;
		}

		// Sort routes by priority
		if (!empty($routerFactories)) {
			krsort($routerFactories, SORT_NUMERIC);
			$routerFactories = array_reverse($routerFactories, true);

			foreach ($routerFactories as $priority => $items) {
				$routerFactories[$priority] = $items;
			}

			// Process all routes services by priority...
			foreach ($routerFactories as $priority => $items) {
				// ...and by service name...
				foreach ($items as $serviceName) {
					$factory = new Nette\DI\Statement(['@' . $serviceName, 'createRouter']);
					$router->addSetup('offsetSet', [null, $factory]);
				}
			}
		}
	}

	public function getTranslationResources()
	{
		$resources   = parent::getTranslationResources();
		$resources[] = APP_DIR . '/lang';

		return $resources;
	}
}
