<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;

class Navigation extends BaseControl
{
	/** @var array */
	protected $data;

	protected array $sections = [
		'top'         => [
			'title' => null,
			'items' => [],
			'order' => 0,
		],
		'eshop'       => [
			'title' => 'core.adminNavigation.eshop',
			'items' => [],
			'order' => 100,
		],
		'pageContent' => [
			'title' => 'core.adminNavigation.pageContent',
			'items' => [],
			'order' => 200,
		],
		'settings'    => [
			'title' => 'core.adminNavigation.settings',
			'items' => [],
			'order' => 300,
		],
	];

	public function render(): void
	{
		$this->template->sections = $this->sections;
		$this->template->render($this->getTemplateFile());
	}

	public function setActiveNav(): void
	{
		$this->setDataToSections($this->data);
		$presenter = $this->presenter;

		if ($presenter instanceof AdminPresenter) {
			$setCurrent = function($navs, $section) use (&$setCurrent, $presenter) {
				/** @var DaoNavigationItem[] $navs */
				foreach ($navs as $nav) {
					$link           = is_array($nav->link) ? $nav->link[0] : $nav->link;
					$nav->isCurrent = $nav->link && (($presenter->isLinkCurrent($link, is_array($nav->link) ? $nav->link[1] : [])) || (!is_array($nav->link) && $presenter->isModuleCurrent($link)));

					if ($nav->isCurrent) {
						$presenter->activeSection    = $section;
						$presenter->activeNavigation = $nav;
					}

					if ($nav->childs) {
						$setCurrent($nav->childs, $section);
					}
				}
			};

			foreach ($this->sections as $section => $navs) {
				$setCurrent($navs['items'], $this->t($navs['title']));
			}
		}
	}

	/**
	 * Data z konfigu
	 *
	 * @param array $data
	 */
	public function setData(array $data): void
	{
		$this->data = $data;
	}

	/**
	 * @return array
	 */
	public function getData(): array
	{
		return $this->data;
	}

	/**
	 * Nastaví data z konfigu do DAO
	 *
	 * @param array $data
	 */
	public function setDataToSections(array $data): void
	{
		foreach ($data as $section => $rows) {
			$this->checkShowIfRecursive($rows);

			foreach ($rows as $k => $row) {
				if ($k === 'sectionConfig') {
					continue;
				}

				if (!isset($row['title'])) {
					continue;
				}

				$nav = new DaoNavigationItem($row);
				if ($this->checkAcl($nav)) {
					if (!isset($this->sections[$section])) {
						$this->sections[$section] = [
								'title' => 'core.adminNavigation.' . $section,
								'items' => [],
							] + ($rows['sectionConfig'] ?? []);
					}

					$this->sections[$section]['items'][] = $nav;
				}
			}
		}

		usort($this->sections, static fn(array $a, array $b) => (int) $a['order'] - (int) $b['order']);

		foreach ($this->sections as $section => $v) {
			$this->sort($this->sections[$section]['items']);
		}
	}

	protected function checkShowIfRecursive(array &$data): void
	{
		$loop = function(array &$rows) use (&$loop): void {
			foreach ($rows as $k => $v) {
				if (!$this->checkShowIf($v)) {
					unset($rows[$k]);
					continue;
				}

				if ($v['items']) {
					$loop($v['items']);
				}
			}
		};

		$loop($data);
	}

	protected function checkShowIf(array $row): bool
	{
		return !(isset($row['showIf']) && !$row['showIf']);
	}

	/**
	 * Zkontroluje jestli má uživatel přístup. Podařazené položky rovnou maže
	 *
	 * @param DaoNavigationItem $nav
	 *
	 * @return bool
	 */
	protected function checkAcl(DaoNavigationItem $nav): bool
	{
		$user = $this->presenter->getUser();

		if ($nav->securityConfig === false) {
			return false;
		}

		$allowed = !$nav->security;

		if (!$allowed) {
			foreach ($nav->security as $m => $ps) {
				if ($user->isAllowed($m, $ps)) {
					$allowed = true;
					break;
				}
			}
		}

		foreach ($nav->childs as $k => $c) {
			if (!$this->checkAcl($c)) {
				$nav->childs[$k]->hidden = true;
			}
		}

		return $allowed;
	}

	/**
	 * Seřadí položky podle order
	 *
	 * @param DaoNavigationItem[] $data
	 *
	 * @return DaoNavigationItem[]
	 */
	protected function sort(&$data): array
	{
		if (is_array($data)) {
			usort($data, function($a, $b) {
				return $a->order - $b->order;
			});

			foreach ($data as &$v) {
				if ($v->childs) {
					$this->sort($v->childs);
				}
			}
		}

		return $data;
	}
}
