<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\PublicHoliday;

use Core\AdminModule\Model\PublicHolidays;
use Core\Model\Countries;
use Core\Model\Entities\PublicHoliday;
use Core\Model\Entities\PublicHolidayText;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Utils\ArrayHash;

class PublicHolidayForm extends BaseControl
{
	public ?PublicHoliday $publicHoliday = null;

	/** @var ?int @persistent */
	public ?int $id = null;

	protected Countries      $countriesService;
	protected PublicHolidays $publicHolidays;

	public function __construct(
		Countries      $countries,
		PublicHolidays $publicHolidays
	)
	{
		$this->countriesService = $countries;
		$this->publicHolidays   = $publicHolidays;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addDatePicker('date', 'admin.publicHoliday.date')
			->setRequired();
		$form->addText('title', 'admin.publicHoliday.title')
			->setIsMultilanguage();
		$form->addTextArea('description', 'admin.publicHoliday.description')
			->setIsMultilanguage();
		$form->addSelect('country', 'admin.publicHoliday.country', ['' => ''] + $this->countriesService->getAllNameColumn());
		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): bool
	{
		$ph   = $this->getPublicHoliday();
		$date = $values->date->format('Y-m-d');

		if (!$ph || $ph->date->format('Y-m-d') !== $date) {
			$exist = $this->publicHolidays->em->getConnection()->fetchOne("SELECT id FROM core__public_holiday WHERE `date` = ?", [$date]);
			if ($exist) {
				$form->getComponent('date')->addError('admin.publicHoliday.dateExist');
				$this->redrawControl('form');

				return false;
			}
		}

		return true;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			$date       = \DateTime::createFromImmutable($values->date);

			/** @var PublicHolidayText[] $texts */
			$texts = [];

			if ($this->id) {
				$ph = $this->publicHolidays->get($this->id);
				if (!$ph) {
					return false;
				}

				$texts = $ph->getTexts()->toArray();
			} else {
				$ph = new PublicHoliday($date);
			}


			$ph->date    = $date;
			$ph->country = $values->country ? $this->countriesService->get($values->country) : null;

			$this->em->persist($ph);

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l])) {
					$texts[$l] = new PublicHolidayText($ph, $l);

					$ph->texts->set($l, $texts[$l]);
				}

				$texts[$l]->title       = $v['title'];
				$texts[$l]->description = $v['description'];
				$this->em->persist($texts[$l]);
			}

			$this->em->persist($ph);
			$this->em->flush();

			$this->em->clear();
			$this->publicHolidays->clearCache();

			$form->addCustomData('publicHolidayId', $ph->getId());
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		$this->redrawControl('form');

		return true;
	}

	public function getPublicHoliday(): ?PublicHoliday
	{
		if ($this->publicHoliday === null && $this->id) {
			$this->publicHoliday = $this->publicHolidays->get($this->id);
		}

		return $this->publicHoliday;
	}

	public function setPublicHoliday(int $id): void
	{
		$this->id = $id;
		$ph       = $this->publicHolidays->get($id);

		if (!$ph) {
			throw new \InvalidArgumentException();
		}

		$this->publicHoliday = $ph;

		$d = [
			'date'    => $ph->date,
			'country' => $ph->country ? $ph->country->getId() : null,
		];

		foreach ($ph->texts->toArray() as $l => $text) {
			$d['title'][$l]       = $text->title;
			$d['description'][$l] = $text->description;
		}

		$this['form']->setDefaults($d);
	}
}
