<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Settings;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\Entities\SiteText;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Utils\ArrayHash;
use Currency\Model\Config as CurrencyConfig;

class SiteForm extends BaseControl
{
	protected string    $ident;
	protected Container $container;
	protected Sites     $sitesService;
	protected Countries $countries;

	public function __construct(
		string    $ident,
		Container $container,
		Sites     $sites,
		Countries $countries
	)
	{
		$this->ident        = $ident;
		$this->container    = $container;
		$this->sitesService = $sites;
		$this->countries    = $countries;

		$this->monitor(AdminPresenter::class, function() {
			$site = $this->sitesService->get($this->ident);
			if ($site) {
				$d = [
					'siteNameSeparator' => $site->siteNameSeparator,
					'email'             => $site->email,
				];

				foreach ($site->texts->toArray() as $l => $v) {
					/** @var SiteText $v */
					$d['domain'][$l]          = $v->domain;
					$d['siteName'][$l]        = $v->siteName;
					$d['logo'][$l]            = $v->logo;
					$d['inverseLogo'][$l]     = $v->inverseLogo;
					$d['isActive'][$l]        = $v->isActive;
					$d['defaultCountry'][$l]  = $v->defaultCountry;
					$d['defaultCurrency'][$l] = $v->defaultCurrency;
				}

				$this['form']->setDefaults($d);
			}
		});
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addBool('isActive', 'admin.site.isActive')
			->setIsMultilanguage();
		$form->addText('domain', 'admin.site.domain')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addFilesManager('logo', 'admin.site.logo')
			->setIsMultilanguage();
		$form->addFilesManager('inverseLogo', 'admin.site.inverseLogo')
			->setIsMultilanguage();
		$form->addText('siteName', 'admin.site.siteName')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('siteNameSeparator', 'admin.site.siteNameSeparator')
			->setNullable()
			->setDefaultValue('|')
			->setMaxLength(255);
		$form->addEmail('email', 'admin.site.email')
			->setMaxLength(255)
			->setRequired();
		$form->addSelect('defaultCountry', 'admin.site.defaultCountry', $this->countries->getAllNameColumn())
			->setIsMultilanguage();

		$hasCurrency = $this->container->hasService('currency.currencies');
		if ($hasCurrency && class_exists('Currency\Model\Config')) {
			$currencies = [];
			foreach (CurrencyConfig::load('whitelist') as $v) {
				$currencies[$v] = $v;
			}

			$form->addSelect('defaultCurrency', 'admin.site.defaultCurrency', $currencies)
				->setIsMultilanguage();
		}

		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			$site       = $this->sitesService->get($this->ident);
			/** @var SiteText[] $texts */
			$texts = $site->texts->toArray();

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l])) {
					$texts[$l] = new SiteText($site, $v['domain'], $l);
				}

				$texts[$l]->domain          = $v['domain'];
				$texts[$l]->logo            = $v['logo'] ?: null;
				$texts[$l]->inverseLogo     = $v['inverseLogo'] ?: null;
				$texts[$l]->siteName        = $v['siteName'];
				$texts[$l]->isActive        = (int) $v['isActive'];
				$texts[$l]->defaultCountry  = $v['defaultCountry'];
				$texts[$l]->defaultCurrency = $v['defaultCurrency'] ?? null;

				$this->em->persist($texts[$l]);
			}

			$site->setTexts($texts);
			$site->siteNameSeparator = $values->siteNameSeparator;
			$site->email             = $values->email;

			$this->em->persist($site);
			$this->em->flush();

			$this->em->getConfiguration()->getResultCacheImpl()->delete('siteTextCache');

			$this->presenter->flashMessageSuccess('admin.site.dataUpdated');
			$this->presenter->redrawControl('flashes');

			$cache = new Cache($this->cacheStorage, \Core\Model\Sites::CACHE_NAMESPACE);
			$cache->remove(\Core\Model\Sites::CACHE_NAMESPACE);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}
}
