<?php declare(strict_types = 1);

namespace Core\AdminModule\Model;

use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\Redirect;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Navigations\AdminModule\Model\Navigations as AdminNavigations;
use Navigations\Model\NavigationConfig;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Tracy\Debugger;

/**
 * @method Redirect|object|null getReference($id)
 * @method Redirect[]|null getAll()
 * @method Redirect|null get($id)
 */
class Redirects extends BaseEntityService
{
	protected $entityClass = Redirect::class;

	protected AdminNavigations  $navigations;
	protected \Core\Model\Sites $sites;

	public function __construct(
		AdminNavigations  $navigations,
		\Core\Model\Sites $sites
	)
	{
		$this->navigations = $navigations;
		$this->sites       = $sites;
	}

	/**
	 * @deprecated
	 */
	public function findOneBy(array $criteria): ?Redirect
	{
		/** @var Redirect|null $redirect */
		$redirect = $this->getEr()->findOneBy($criteria);

		if ($redirect) {
			$redirect->views++;
			$this->em->persist($redirect);
			$this->em->flush();
		}

		return $redirect;
	}

	public function removeRedirects(array $ids): bool
	{
		try {
			$this->getEr()->createQueryBuilder('r')
				->delete()
				->where('r.id IN (' . implode(',', $ids) . ')')
				->getQuery()->execute();
			$cache = new Cache($this->cacheStorage, Redirect::CACHE_NAMESPACE);
			$cache->clean([Cache::All => true]);

			return true;
		} catch (\Exception $e) {
			Debugger::log($e);
		}

		return false;
	}

	public function validateRedirects(int $offset, int $limit): void
	{
		foreach ($this->getEr()->createQueryBuilder('r')
			         ->where('r.lastCheck IS NULL OR r.lastCheck < :date')
			         ->setParameter('date', (new DateTime())->modify('-1 week'))
			         ->setMaxResults($limit)
			         ->setFirstResult($offset)
			         ->getQuery()->getArrayResult() as $row) {
			if (Strings::startsWith($row['to'], 'http')) {
				continue;
			}

			// Validace dat
			if ($row['siteIdent'] === null) {
				if ($row['relationKey'] === 'Navigation') {
					$nav = $this->navigations->get((int) $row['relationValue']);
					if ($nav) {
						/** @var Redirect $redirect */
						$redirect            = $this->getReference($row['id']);
						$redirect->siteIdent = $nav->getSite()->getIdent();
						$redirect->lang      = $nav->locale;
						$this->em->persist($redirect);

						$row['siteIdent'] = $redirect->siteIdent;
						$row['lang']      = $redirect->lang;
					}
				}
			}

			// Kontrola presmerovani
			$today     = (new DateTime())->format('Y-m-d');
			$responses = [];

			if ($row['siteIdent'] && $row['lang']) {
				$site = $this->sites->getSites()[$row['siteIdent']];

				if ($site) {
					$domain = $site->getDomains()[$row['lang']];

					if ($domain) {
						$responses[] = $this->getResponse($this->buildUrl($domain, $row['to']));
					}
				}
			} else {
				foreach ($this->sites->getSites() as $site) {
					foreach ($site->getDomains() as $domain) {
						if (!$domain->getDomain()) {
							continue;
						}

						$tmp = $this->getResponse($this->buildUrl($domain, $row['to']));
						if ($tmp['httpCode'] == '200') {
							$responses[] = $tmp + [
									'siteIdent' => $site->getIdent(),
									'lang'      => $domain->getLang(),
								];
						}
					}
				}

				if (empty($responses)) {
					$responses[] = [
						'error'    => '',
						'httpCode' => '404',
					];
				}
			}

			foreach ($responses as $k => $response) {
				/** @var Redirect $redirect */
				$redirect = $this->getReference($row['id']);

				if ($k > 0) {
					$redirect = clone $redirect;
				}

				if ($response['error']) {
					FileSystem::createDir(LOG_DIR . '/_redirects');
					Debugger::log('VALIDATE ERROR - ' . $row['id'] . ' - ' . $response['error'], '_redirects/' . $today . '.log');
					$redirect->updateCheck('validate error');
				} else {
					$redirect->updateCheck($response['httpCode']);
				}

				if (isset($response['siteIdent'])) {
					$redirect->siteIdent = $response['siteIdent'];
				}

				if (isset($response['lang'])) {
					$redirect->lang = $response['lang'];
				}

				$this->em->persist($redirect);
			}

			$this->em->flush();
			sleep(1);
		}

		$cache = new Cache($this->cacheStorage, Redirect::CACHE_NAMESPACE);
		$cache->clean([Cache::All => true]);
	}

	protected function buildUrl(SiteDomain $domain, string $to): string
	{
		$url = 'https://' . $domain->getDomain();
		if (!$domain->isDefault) {
			$url .= '/' . (NavigationConfig::load('useCZinUrl') === true && $domain->getLang() == 'cs' ? 'cz' : $domain->getLang());
		}

		$url .= '/' . ltrim($to, '/');

		return $url;
	}

	protected function getResponse(string $url): array
	{
		$ch = curl_init($url);

		if (!$ch) {
			Debugger::log("CURL init failed $url");
			throw new \Exception("CURL init failed $url");
		}

		curl_setopt($ch, CURLOPT_HEADER, true);
		curl_setopt($ch, CURLOPT_NOBODY, true);
		curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($ch, CURLOPT_TIMEOUT, 10);
		$output   = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$error    = curl_error($ch);
		curl_close($ch);

		return [
			'error'    => $error,
			'httpCode' => (string) $httpCode,
		];
	}
}
