<?php declare(strict_types = 1);

namespace Core\Console\Seo;

use Contributte\Translation\Translator;
use Core\FrontModule\Model\SiteMap;
use Core\Model\Dao\Site;
use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Nette\Http\Request;
use Nette\Utils\FileSystem;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Thepixeldeveloper\Sitemap\Drivers\XmlWriterDriver;
use Tracy\Debugger;

class GenerateSiteMap extends AbstractCommand
{
	/** @var SiteMap @inject */
	public $siteMap;

	/** @var Request @inject */
	public $httpRequest;

	/** @var Langs @inject */
	public $langs;

	/** @var Sites @inject */
	public ?Sites $sites = null;

	/** @var Translator @inject */
	public ?Translator $translator = null;

	protected function configure(): void
	{
		$this->setName('seo:generate-sitemap')
			->addOption('site', null, InputOption::VALUE_OPTIONAL)
			->addOption('lang', null, InputOption::VALUE_OPTIONAL)
			->setDescription('Generate sitemaps');
	}

	protected function executeSchemaCommand(
		InputInterface                $input,
		OutputInterface               $output,
		Doctrine\ORM\Tools\SchemaTool $schemaTool,
		array                         $metadatas,
		SymfonyStyle                  $ui
	)
	{
		$output->writeln('GENERATE SITEMAP.XML');

		try {
			set_time_limit(3000);
			ini_set('memory_limit', '2G');
			Debugger::dispatch();
			$groups    = ['default'];
			$siteIdent = $input->getOption('site');
			$langValue = $input->getOption('lang');

			$created = [];
			$exist   = [];

			foreach ($this->sites->getSites() as $k => $v) {
				if ($siteIdent && $siteIdent !== $k) {
					continue;
				}

				FileSystem::delete(SiteMap::TMP_DIR . '/' . $k);
				foreach ($v->getDomains() as $domain) {
					if ($langValue && $langValue !== $domain->getLang()) {
						continue;
					}

					$lang                                       = $domain->getLang();
					/** @phpstan-ignore-next-line */
					Sites::$currentIdentOverride                = $v->getIdent();
					Sites::$currentLangOverride                 = $lang;
					$this->sites->getCurrentSite()->currentLang = $lang;
					$this->translator->setLocale($lang);

					foreach ($groups as $group) {
						$sitemap = $this->siteMap->getSiteMap($lang, $group);

						if (!$sitemap) {
							continue;
						}

						$driver = new XmlWriterDriver();
						$sitemap->accept($driver);

						$filename = "sitemap-$k-$group-$lang.xml";

						$filePath = WWW_DIR . '/' . $filename;

						$created[] = $filePath;
						file_put_contents($filePath, $driver->output());

						if (isset($sitemap->all()[0]) && $sitemap->all()[0] instanceof \Thepixeldeveloper\Sitemap\Sitemap) {
							foreach ($sitemap->all() as $sitemapK => $v) {
								/** @var \Thepixeldeveloper\Sitemap\Sitemap $v */
								$tmp = $this->siteMap->getSiteMap($lang, $group, $sitemapK + 1);

								if (!$tmp) {
									continue;
								}

								$driver = new XmlWriterDriver();
								$tmp->accept($driver);

								$filePath  = WWW_DIR . '/' . basename($v->getLoc());
								$created[] = $filePath;
								file_put_contents($filePath, $driver->output());
								$tmp = null;
							}
						}
						$sitemap = null;
					}
				}
			}

			$output->writeLn('<info>[OK] - sitemaps created</info>');

			return 0;
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
