<?php declare(strict_types = 1);

namespace Core\FrontModule\Model;

use Core\Model\Entities\Seo;
use Core\Model\Helpers\Strings;
use Core\Model\SystemConfig;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\Controls\TextInput;
use Nette\Utils\ArrayHash;

class SeoContainer
{
	protected SiteMap $siteMapService;

	public function __construct(
		SiteMap $siteMap
	)
	{
		$this->siteMapService = $siteMap;
	}

	public function getContainer(bool $multilang = false): BaseContainer
	{
		$container = new BaseContainer();

		$container->addText('title', 'default.seoContainer.title')->setDescription('default.seoContainer.titleDesc');
		$description = $container->addTextArea('description', 'default.seoContainer.description')
			->setDescription('default.seoContainer.titleDesc');

		if (SystemConfig::load('metaDescriptionMaxLength')) {
			$description->setMaxLength((int) SystemConfig::load('metaDescriptionMaxLength'));
		}

		$container->addSelect('robots', 'default.seoContainer.robots', [
			'index, follow'     => 'index, follow',
			'noindex, follow'   => 'noindex, follow',
			'noindex, nofollow' => 'noindex, nofollow',
		]);
		$container->addText('canonical', 'default.seoContainer.canonical')->setPlaceholder('self');
		$container->addBool('addToSiteMap', 'default.seoContainer.addToSiteMap')->setDefaultValue(1);
		$container->addSelect('siteMapChangeFreq', 'default.seoContainer.siteMapChangeFreq', $this->siteMapService->getAllChangeFreq())
			->setDefaultValue(SiteMap::CHANGE_FREQ_MONTHLY);
		$container->addText('siteMapPriority', 'default.seoContainer.siteMapPriority')->setDefaultValue('0.5');

		if ($multilang) {
			foreach ([
				         'title',
				         'description',
				         'robots',
				         'canonical',
				         'addToSiteMap',
				         'siteMapChangeFreq',
				         'siteMapPriority',
			         ] as $k) {
				/** @var TextInput $component */
				$component = $container->getComponent($k);
				$component->setIsMultilanguage();
			}
		}

		return $container;
	}

	/**
	 * @param array|ArrayHash $values
	 */
	public function getFormData($values, bool $multilang = false): array
	{
		if ($multilang) {
			$return = [];

			foreach ((array) $values as $key => $value) {
				foreach ($value as $lang => $v) {
					$return[$lang][$key] = $v;
				}
			}

			return $return;
		}

		return (array) $values;
	}

	public function setDefaults(BaseContainer $container, array $values): void
	{
		if (isset($values['description']) && SystemConfig::load('metaDescriptionMaxLength')) {
			if (is_array($values['description'])) {
				foreach ($values['description'] as $k => $v) {
					$values['description'][$k] = Strings::truncate((string) $v, (int) SystemConfig::load('metaDescriptionMaxLength'), '');
				}
			} else {
				$values['description'] = Strings::truncate((string) $values['description'], (int) SystemConfig::load('metaDescriptionMaxLength'), '');
			}
		}

		foreach ($container->getControls() as $c) {
			if (isset($values[$c->getName()])) {
				$c->setDefaultValue($values[$c->getName()]);
			}
		}
	}

	public function setDefaultsFromEntity(BaseContainer $container, array $texts): void
	{
		$seoData = [];
		foreach ($texts as $l => $data) {
			if (is_object($data) && method_exists($data, 'getSeo')) {
				foreach ($data->getSeo() as $k => $v) {
					$seoData[$k][$l] = $v;
				}
			} else if (property_exists($data, 'seo') && $data->seo instanceof Seo) {
				$seoData['title'][$l]             = $data->seo->title;
				$seoData['description'][$l]       = $data->seo->description;
				$seoData['robots'][$l]            = $data->seo->robots;
				$seoData['canonical'][$l]         = $data->seo->canonical;
				$seoData['addToSiteMap'][$l]      = $data->seo->addToSiteMap;
				$seoData['siteMapChangeFreq'][$l] = $data->seo->siteMapChangeFreq;
				$seoData['siteMapPriority'][$l]   = $data->seo->siteMapPriority;
			}
		}

		$this->setDefaults($container, $seoData);
	}
}
