<?php declare(strict_types = 1);

namespace Core\Model\Helpers;

use Core\Model\Lang\Langs;
use Doctrine\Common\Collections\Collection;
use Nette\Forms\Container;
use Nette\Forms\Controls\SelectBox;
use Contributte\Translation\Translator;
use Nette\Utils\ArrayHash;

class FormHelper
{
	protected Langs      $langs;
	protected Translator $translator;

	public function __construct(
		Langs      $langs,
		Translator $translator
	)
	{
		$this->langs      = $langs;
		$this->translator = $translator;
	}

	public function getDefaultByTranslateMultiLang(string $text): array
	{
		$default = [];

		foreach ($this->langs->getLangs(false) as $lang) {
			$tag           = $lang->getTag();
			$default[$tag] = $this->translator->translate($text, [], null, $tag);
		}

		return $default;
	}

	public static function fillContainerByEntity(Container $container, object $entity, array $skip = []): void
	{
		foreach ($container->getControls() as $input) {
			$name = $input->getName();

			if (!isset($entity->$name) || $entity->$name instanceof Collection || Arrays::contains($skip, $name)) {
				continue;
			}

			$value = $entity->$name;
			if (is_object($value)) {
				if (method_exists($value, 'getId')) {
					$value = $value->getId();
				} else if (method_exists($value, 'getIdent')) {
					$value = $value->getIdent();
				}
			}

			if ($input instanceof SelectBox) {
				if (array_key_exists($value, $input->getItems())) {
					$input->setDefaultValue($value);
				}
			} else {
				$input->setDefaultValue($value);
			}
		}
	}

	/**
	 * @param ArrayHash|array $values
	 */
	public static function fillEntityByValues(object $entity, $values): void
	{
		foreach (get_object_vars($entity) as $k => $v) {
			if (
				(is_object($values) && !property_exists($values, $k))
				|| (is_array($values) && !array_key_exists($k, $values)
				)) {
				continue;
			}

			if ($values instanceof ArrayHash) {
				$values = (array) $values;
			}

			if (isset($values[$k]) || $values[$k] === null) {
				$entity->$k = $values[$k];
			}
		}
	}

	public static function getTemplateFiles(string $subDirectory): array
	{
		$arr = [];

		$subDirArr = explode('/', $subDirectory);

		foreach (glob(TEMPLATES_DIR . DS . 'Front' . DS . '*', GLOB_ONLYDIR) ?: [] as $dir) {
			$dirName = basename($dir);

			foreach (glob($dir . DS . $subDirectory . DS . '*.latte') ?: [] as $file) {
				$fileName = pathinfo($file, PATHINFO_FILENAME);

				$arr[$dirName][$dirName . '|' . $fileName] = 'admin.' . implode('.', array_map(static fn($v) => lcfirst($v), $subDirArr)) . '.' . $fileName;
			}
		}

		if (count($arr) === 1) {
			$arr = array_values($arr)[0];
		}

		return $arr;
	}
}
