<?php declare(strict_types = 1);

namespace Core\Model\TemplateTextType;

use Contributte\Translation\Translator;
use Core\Model\Helpers\Arrays;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\Navigations as FrontNavigation;
use Nette\Application\LinkGenerator;
use Nette\DI\Container;
use Nette\Utils\Json;

class Link extends TemplateTextType implements ITemplateTextType
{
	/** @var string */
	protected $type = 'link';

	protected Container     $container;
	protected Translator    $translator;
	protected LinkGenerator $linkGenerator;

	public function __construct(
		Container     $container,
		Translator    $translator,
		LinkGenerator $linkGenerator
	)
	{
		$this->container     = $container;
		$this->translator    = $translator;
		$this->linkGenerator = $linkGenerator;
	}

	/**
	 * @param BaseContainer $formContainer
	 * @param array         $params
	 */
	public function loadContainer($formContainer, $params = [], $texts = []): void
	{
		$eshopExist = class_exists('EshopCatalog\DI\EshopCatalogExtension');

		$container = $formContainer->addContainer($this->getName(), $this->getTitle());
		$container->addCustomData('multiLang', true);
		$container->addCustomData('renderFile', __DIR__ . '/Link.latte');

		$types = [
			''      => '',
			'text'  => 'admin.ttType.link.typeList.text',
			'phone' => 'admin.ttType.link.typeList.phone',
			'email' => 'admin.ttType.link.typeList.email',
			'file'  => 'admin.ttType.link.typeList.file',
		];

		if ($eshopExist) {
			$types['product']  = 'eshopCatalog.ttType.link.typeList.product';
			$types['category'] = 'eshopCatalog.ttType.link.typeList.category';
		}

		if ($this->container->hasService('navigations.admin.navigations')) {
			$types['navigation'] = 'admin.ttType.link.typeList.navigation';
		}

		$container->addSelect('type', 'admin.ttType.link.type', $types)
			->getControlPrototype()->setAttribute('data-tt-link-select', $this->getName());
		$container->addBool('openInNewTab', 'admin.ttType.link.openInNewTab')
			->setDefaultValue(0);

		$typesContainer = $container->addContainer('types', 'admin.ttType.link.typesContainer');
		$typesContainer->addCustomData('ttType', $this->getName());
		$typesContainer->addCustomData('multiLang', true);

		$typesContainer->addText('text', 'admin.ttType.link.text');
		$typesContainer->addText('phone', 'admin.ttType.link.phone');
		$typesContainer->addText('email', 'admin.ttType.link.email');
		$typesContainer->addFilesManager('file', 'admin.ttType.link.file');

		if ($eshopExist) {
			$container->addHidden('productId');
			$typesContainer->addText('product', 'eshopCatalog.ttType.product')
				->setHtmlAttribute('data-autocomplete-name', 'ttLink-product')
				->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll'))
				->setHtmlAttribute('data-autocomplete-target', $container->getComponent('productId')->getHtmlId())
				->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name');
			$container->addHidden('categoryId');
			$typesContainer->addText('category', 'eshopCatalog.ttType.category')
				->setHtmlAttribute('data-autocomplete-name', 'ttLink-category')
				->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Categories:categories'))
				->setHtmlAttribute('data-autocomplete-target', $container->getComponent('categoryId')->getHtmlId())
				->setHtmlAttribute('data-autocomplete-keys', 'id,name');
		}

		if ((bool) $params['multiLang'] === true) {
			$typesContainer->getComponent('text')->setIsMultilanguage();
			$typesContainer->getComponent('phone')->setIsMultilanguage();
			$typesContainer->getComponent('email')->setIsMultilanguage();
			$typesContainer->getComponent('file')->setIsMultilanguage();
		}

		if ($this->container->hasService('navigations.admin.navigations')) {
			/** @var Navigations $navigations */
			$navigations = $this->container->getService('navigations.admin.navigations');

			$typesContainer->addSelect('navigation', 'admin.ttType.link.navigation', ['' => ''] + $navigations->getOptionsForSelect());

			if ($params['multiLang'] === true) {
				$typesContainer->getComponent('navigation')->setIsMultilanguage();
			}
		}
	}

	public function render(array $params): array
	{
		$eshopExist = class_exists('EshopCatalog\DI\EshopCatalogExtension');
		$data       = $this->getDefault();

		if (is_string($data) && Arrays::isJson($data)) {
			$data = Json::decode($data, Json::FORCE_ARRAY);
		}

		if ($data['type'] === '') {
			return [];
		}

		$lang     = $params['multiLang'] ? $this->translator->getLocale() : null;
		$getValue = static function($v) use ($lang) {
			if (is_array($v) && array_key_exists($lang, $v)) {
				$v = $v[$lang];
			}

			return $v;
		};

		$link   = null;
		$result = [
			'type'   => $data['type'],
			'target' => (int) $data['openInNewTab'] === 1 ? '_blank' : '_self',
			'value'  => $getValue($data['types'][$data['type']]),
		];

		switch ($data['type']) {
			case 'phone':
				$link = 'tel:' . str_replace(' ', '', (string) $getValue($data['types']['phone']));
				break;
			case 'email':
				$link = 'mailto:' . str_replace(' ', '', (string) $getValue($data['types']['email']));
				break;
			case 'navigation':
				if ($this->container->hasService('navigations.navigations')) {
					/** @var FrontNavigation $navigations */
					$navigations = $this->container->getService('navigations.navigations');
					$nav         = $navigations->getNavigation((int) $getValue($data['types']['navigation']));

					if ($nav) {
						$link                 = $nav->link;
						$result['navigation'] = $nav;
					}
				}
				break;
			case 'product':
				if ($eshopExist && class_exists(ProductsFacade::class) && $this->container->hasService('eshopCatalog.front.productsFacade')) {
					/** @var ProductsFacade $productsFacade */
					$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');
					$productId      = $getValue($data['productId']);

					$product = $productId ? $productsFacade->getProduct((int) $productId) : null;
					if (!$product) {
						break;
					}

					$link              = $product->link;
					$result['product'] = $product;
				}
				break;
			case 'category':
				if ($eshopExist && class_exists(Categories::class) && $this->container->hasService('eshopCatalog.front.categories')) {
					/** @var Categories $categories */
					$categories = $this->container->getService('eshopCatalog.front.categories');
					$categoryId = $getValue($data['categoryId']);

					$cat = $categoryId ? $categories->get((int) $categoryId) : null;
					if (!$cat) {
						break;
					}

					$link               = $cat->link;
					$result['category'] = $cat;
				}
				break;
			default:
				$link = $getValue($data['types'][$data['type']]);
		}

		$result['link'] = $link;

		return $result;
	}
}
