<?php declare(strict_types = 1);

namespace Core\Model\Translation;

use DeepL\Translator;
use DeepL\Usage;

class DeepLApi
{
	protected string $authKey;
	protected array  $locales = [
		'cs' => 'cs',
		'en' => 'en-GB',
		'es' => 'es',
		'uk' => 'uk',
	];

	public function __construct(
		string $authKey
	)
	{
		$this->authKey = $authKey;
	}

	public function translateString(string $str, string $sourceLang, string $targetLang): string
	{
		if ($str === strip_tags($str)) {
			return $this->translateText($str, $sourceLang, $targetLang);
		}

		return $this->translateHtml($str, $sourceLang, $targetLang);
	}

	public function translateText(string $str, string $sourceLang, string $targetLang): string
	{
		if ($str === '') {
			return $str;
		}

		$translator = new Translator($this->authKey);

		$data = $translator->translateText($str, $this->locales[$sourceLang], $this->locales[$targetLang], [
			'split_sentences'        => 1,
			'preserve_formatting'    => true,
			'show_billed_characters' => true,
		]);

		return $data->text;
	}

	public function translateHtml(string $html, string $sourceLang, string $targetLang): string
	{
		if ($html === '' || strip_tags($html) === '') {
			return $html;
		}

		$translator = new Translator($this->authKey);

		$html = str_replace("\n", '||||', $html);

		$data = $translator->translateText($html, $this->locales[$sourceLang], $this->locales[$targetLang], [
			'tag_handling'           => 'html',
			'show_billed_characters' => true,
		]);

		$result = $data->text;

		$result = str_replace('||||', "\n", $result);

		return $result;
	}

	public function getUsage(): Usage
	{
		$translator = new Translator($this->authKey);

		return $translator->getUsage();
	}
}
