<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\AdminModule\Presenters\BasePresenter;
use Core\Components\FilesManager\FilesManager as FilesManagerComponent;
use Core\Model\FilesManager\FilesManager;
use Core\Model\Helpers\Strings;
use Core\Model\SystemConfig;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Traits\TInputExtend;
use Core\Model\UI\Form\Traits\TTextBase;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Utils\Arrays;
use Nette\Utils\Html;
use Nette\Utils\Json;

/**
 * @method BaseForm getForm(bool $throw = true)
 */
class FilesManagerInput extends \Nette\Forms\Controls\TextInput
{
	use TInputExtend;
	use TTextBase;

	public FilesManager              $filesManager;
	protected ?FilesManagerComponent $fmComponent = null;

	protected array $securedRoots   = [];
	public bool     $multiple       = false;
	private ?string $controllerName = null;

	/**
	 * @param string|object $caption
	 */
	public function __construct($caption, FilesManager $filesManager, Session $session, Request $request)
	{
		parent::__construct($caption);
		$this->filesManager = $filesManager;

		$this->monitor(BaseForm::class, function() {
			$currGroup = $this->getForm()->getCurrentGroup();

			$this->getForm()->setCurrentGroup(null);
			$this->getForm()->setCurrentGroup($currGroup);
		});

		$this->monitor(BasePresenter::class, function(BasePresenter $presenter) use ($session, $request) {
			// Pokud existuje nejake nastaveni, nutno volat novou komponentu
			if ($this->securedRoots) {
				$this->fmComponent = new FilesManagerComponent($this->filesManager, $session, $request);
				$this->fmComponent->addSecuredRoot($this->securedRoots);
				$this->attachController();
			} else {
				$this->controllerName = 'filesManager';
				$this->fmComponent    = $presenter['filesManager'];
			}
		});
	}

	protected function attachController(): void
	{
		$name = 'filesManagerController';
		foreach (explode('-', Strings::webalize($this->getHtmlName())) as $v) {
			$name .= ucfirst($v);
		}

		$this->controllerName = $name;
		if (!$this->form->getComponent($this->controllerName, false)) {
			$this->form->addComponent($this->fmComponent, $this->controllerName);
		}
	}

	public function addSecuredRoot(array $roots): self
	{
		$this->securedRoots = array_merge($this->securedRoots, $roots);

		return $this;
	}

	public function getControl(?Html $content = null): Html
	{
		$this->fmComponent->setModal();
		$langs       = $this->getForm()->langsService->getLangs(false);
		$isMultilang = $this->getIsMultiLanguage();
		$mainWrap    = Html::el();
		$defaultLang  = array_key_first($langs);
		$selectedLang = $_COOKIE['contentLang'] ?? $defaultLang;
		if (!array_key_exists($selectedLang, $langs)) {
			$selectedLang = $defaultLang;
		}

		foreach ($isMultilang ? $langs : ['' => null] as $k => $lang) {
			$input   = parent::getControl();
			$inputId = $this->getHtmlId() . ($k ? '-' . $k : '');

			$wrapper = Html::el('div', [
				'class'         => 'filemanager-control',
				'data-base-id'  => $this->getHtmlId(),
				'data-input-id' => $inputId,
			]);

			$input->attrs['type'] = 'hidden';

			if ($k) {
				$wrapper->attrs['data-content-lang'] = $k;
				$input->attrs['name']                .= "[$k]";

				if ($k !== $selectedLang) {
					$wrapper->attrs['style'] = 'display: none;';
				}
			}

			$input->attrs['id'] = $inputId;

			$btnThumbnail = Html::el('a', [
				'href'   => 'javascript:;',
				'target' => '_blank',
				'class'  => 'btn btn-info filesManagerInput-thumbnail',
				'id'     => $inputId . '-thumbnail',
			])->addHtml('<i class="fa fa-file"></i>');

			$fakeInput = Html::el('input', [
				'type'     => 'text',
				'readonly' => 'true',
				'class'    => 'form-control filesManagerInput-fakeInput',
				'id'       => $inputId . '-fake-input',
			]);

			$btns = Html::el('div', ['class' => 'btn-group']);

			$btn = Html::el('a', [
				'href'  => $this->fmComponent->link('showModal!', [$this->controllerName]),
				'class' => 'btn btn-success filesManagerInput-showModal',
				'id'    => $inputId . '-show-modal',
			])->setHtml($content === null ? $this->translate('default.select') : $content);

			$btnDelete = Html::el('a', [
				'href'  => '#',
				'class' => 'btn btn-danger btn-remove-image-' . $inputId,
			])->addHtml('<i class="fa fa-times"></i>');

			$btns->addHtml($btnThumbnail);
			$btns->addHtml($fakeInput);
			$btns->addHtml($btn);
			$btns->addHtml($btnDelete);

			ob_start();
			$this->fmComponent->render();
			$htmlContent = ob_get_clean();

			$volumes = [];
			$i       = 1;
			foreach (SystemConfig::load('filesManager.roots', []) as $root) {
				if (!isset($root['path'])) {
					continue;
				}

				$path = trim(str_replace('\\', '/', $root['path']), '/\\');
				if (!Strings::contains($path, '/')) {
					continue;
				}

				$parts = explode('/', $path);
				if (!is_array($parts) || count($parts) < 1 || strlen(Arrays::last($parts)) < 1) {
					continue;
				}

				$volume           = Arrays::last($parts);
				$volumes[$volume] = "l{$i}_";
				$i++;
			}

			$wrapper->addHtml($btns);
			$wrapper->addHtml((string) $htmlContent);

			$latte = file_get_contents(__DIR__ . '/FileManagerInput.latte');
			if ($latte) {
				$latte = str_replace('{$inputId}', $inputId, $latte);
				$latte = str_replace('{$baseId}', $this->getHtmlId(), $latte);
				$latte = str_replace('{$controllerName}', $this->controllerName, $latte);
				$latte = str_replace('{$isMultiple}', $this->multiple ? 'true' : 'false', $latte);
				$latte = str_replace('{$volumes}', Json::encode($volumes), $latte);
				$wrapper->addHtml($latte);
			}

			$wrapper->addHtml($input);

			if ($this->getValue()) {
				$value = $this->getValue();
				if (is_array($value) && $this->getIsMultiLanguage()) {
					$value = $value[$k] ?? null;
				}

				$btnThumbnail->attrs['href'] = $value;
				$fakeInput->attrs['value']   = $value;
				$input->attrs['value']       = $value;
			} else {
				$btnDelete->style = ' display: none;';
			}

			$mainWrap->addHtml($wrapper);
		}

		return $mainWrap;
	}

	public static function register(\Nette\DI\Container $systemContainer): void
	{
		$class = __CLASS__;
		\Nette\Forms\Container::extensionMethod("addFilesManager", static function(
			\Nette\Forms\Container $container, $name, $caption
		) use ($class, $systemContainer) {
			/** @var FilesManagerInput $comp */
			$comp = new $class($caption,
				$systemContainer->getByType(FilesManager::class, false),
				$systemContainer->getByType(Session::class, false),
				$systemContainer->getByType(Request::class, false));

			$container->addComponent($comp, $name);

			return $comp;
		});
	}
}
