<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Model\Turnstile\Turnstile;
use Nette\Forms\Controls\TextInput;
use Nette\Forms\Form;
use Nette\Forms\Rules;
use Nette\Utils\Html;

class TurnstileInput extends TextInput
{
	public const sizeNormal  = 'normal';
	public const sizeCompact = 'compact';

	public const themeAuto  = 'auto';
	public const themeLight = 'light';
	public const themeDark  = 'dark';

	protected Turnstile $provider;

	protected bool    $configured = false;
	protected ?string $message    = null;
	protected string  $size       = self::sizeNormal;
	protected string  $theme      = self::themeAuto;
	protected string  $publicKey;

	public function __construct(
		Turnstile $provider,
		?string   $label = null,
		?string   $message = null,
		?string   $publicKey = null
	)
	{
		parent::__construct($label);

		$this->publicKey = $publicKey;
		$this->provider  = $provider;

		$this->setOmitted(true);
		$this->control = Html::el('div');
		$this->control->addClass('cf-turnstile');

		$this->message = $message;
	}

	public function loadHttpData(): void
	{
		$this->setValue($this->getForm()->getHttpData(Form::DATA_TEXT, 'cf-turnstile-response'));
	}

	public function setMessage(string $message): self
	{
		$this->message = $message;

		return $this;
	}

	public function setSize(string $size): self
	{
		$this->size = $size;

		return $this;
	}

	public function setTheme(string $theme): self
	{
		$this->theme = $theme;

		return $this;
	}

	public function validate(): void
	{
		$this->configureValidation();
		parent::validate();
	}

	public function getRules(): Rules
	{
		$this->configureValidation();

		return parent::getRules();
	}

	private function configureValidation(): void
	{
		if ($this->configured) {
			return;
		}

		$this->configured = true;
		$message          = $this->message ?? 'Are you a bot?';
		$this->addRule(function($code): bool {
			return $this->verify() === true;
		}, $message);
	}

	public function verify(): bool
	{
		return $this->provider->validateControl($this) === true;
	}

	public function getControl(): Html
	{
		$this->configureValidation();

		$el = parent::getControl();
		$el->addAttributes([
			'id'           => $this->getHtmlId(),
			'name'         => $this->getHtmlName(),
			'data-sitekey' => $this->publicKey,
			'data-size'    => $this->size,
			'data-theme'   => $this->theme,
		]);

		return $el;
	}

}
