<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Traits;

use Core\Model\Lang\Lang;
use Core\Model\Module;
use Core\Model\Parameters;
use Nette\Forms\Controls\TextArea;
use Nette\Forms\Controls\TextInput;
use Nette\Forms\Helpers;
use Nette\Utils\Html;
use Nette\Utils\Json;

trait TInputExtend
{
	public ?string $description     = null;
	public bool    $isMultiLanguage = false;

	public function setDescription(?string $description = null): self
	{
		$this->description = $description;

		return $this;
	}

	public function getLabel($caption = null): Html
	{
		$label = parent::getLabel($caption);

		if (Module::isAdmin()) {
			$langs        = $this->getForm()->langsService->getLangs(false);
			$hasMoreLangs = count($langs) > 1;

			if ($hasMoreLangs) {
				$wrap = Html::el('div', [
					'class'     => 'label-wrap',
					'data-name' => $this->getHtmlName(),
				]);
				/** @var Lang $defaultLang */
				$defaultLang = array_shift($langs);

				$selectedLang = $_COOKIE['contentLang'] ?? $defaultLang->getTag();
				if (!array_key_exists($selectedLang, $langs)) {
					$selectedLang = $defaultLang->getTag();
				}

				if ($this->getIsMultiLanguage()) {
					$left = Html::el('div');
					$left->addHtml(Html::el('i class="flag-icon" data-flag="' . $selectedLang . '"'));
					$left->addHtml($label);

					$right = Html::el('div', [
						'class' => 'btn-group',
					]);

					if (
						Parameters::load('deepL.enable')
						&& (
							$this instanceof TextInput
							|| $this instanceof TextArea /** @phpstan-ignore-line */
						)) {
						$value   = $this->getValue();
						$tooltip = $this->translate('default.translateTo') . ': ';

						foreach ($langs as $lang) {
							$class = ['btn', 'btn-xs'];

							if ($value[$lang->getTag()] ?? null) {
								$class[] = 'btn-success';
							}

							$right->addHtml(Html::el('a', [
								'class'             => implode(' ', $class),
								'href'              => 'javascript:;',
								'data-translate-to' => $lang->getTag(),
								'title'             => $tooltip . $lang->getTitle(),
								'tabindex'          => -1,
							])->setText($lang->getTag()));
						}
					}

					$wrap->addHtml($left);
					$wrap->addHtml($right);
					$wrap->addClass('label-wrap--multi-lang');

					return $wrap;
				} else if ($this->getForm()->getShowLangSwitcher()) {
					$wrap->addHtml(Html::el('i class="fas fa-globe"'));
					$wrap->addHtml($label);

					return $wrap;
				}
			}
		}

		if (!$label instanceof Html) {
			$label = Html::el()->setHtml($label);
		}

		return $label;
	}

	/**
	 * @param mixed $val
	 */
	public function setIsMultilanguage($val = true): self
	{
		$this->isMultiLanguage = (bool) $val;

		return $this;
	}

	/**
	 * @return mixed
	 */
	public function getDescription() { return $this->getTranslator()->translate($this->description); }

	public function getIsMultiLanguage(): bool { return $this->isMultiLanguage; }

	protected function createTranslatableControl(Html $control): Html
	{
		if ($this->getIsMultiLanguage()) {
			$wrap  = Html::el('div');
			$langs = $this->getForm()->langsService->getLangs();
			/** @var Lang $defaultLang */
			$defaultLang = array_shift($langs);

			$selectedLang = $_COOKIE['contentLang'] ?? $defaultLang->getTag();
			if (!array_key_exists($selectedLang, $langs)) {
				$selectedLang = $defaultLang;
			}

			foreach ($langs as $k => $lang) {
				$c = clone $control;

				// TODO předělat a upravit taky aby bylo povinné pouze jedno pole
				if ($defaultLang->getTag() != $lang->getTag()) {
					unset($c->attrs['required']);

					foreach ($c->attrs['data-nette-rules'] as $k2 => $rule) {
						if ($rule['op'] === ':filled') {
							unset($c->attrs['data-nette-rules'][$k2]);
						}

						break;
					}
				}

				if (property_exists($this, 'placeholder') && is_array($this->placeholder) && array_key_exists($lang->getTag(), $this->placeholder)) {
					$c->attrs['placeholder'] = $this->placeholder[$lang->getTag()];
				}

				$c->attrs['name']              .= "[$k]";
				$c->attrs['id']                .= "-$k";
				$c->attrs['data-content-lang'] = $k;

				if ($k !== $selectedLang) {
					$c->attrs['style'] = 'display: none;';
				}

				$getValue = $this->getValue();
				$value    = (is_array($getValue) ? $getValue[$k] : $getValue->$k) ?? '';

				if ($c->getName() === 'textarea') {
					$c->setText($value);
				} else {
					$c->attrs['value'] = $value;
				}

				$wrap->addHtml($c);
			}

			return $wrap;
		}

		return $control;
	}

	/**
	 * @param int $type
	 *
	 * @return mixed
	 */
	protected function getHttpData($type, string $htmlTail = null)
	{
		if ($this->getIsMultiLanguage()) {
			return array_map(function($lang) use ($type, $htmlTail) {
				return $this->getForm()->getHttpData($type, $this->getHtmlName() . "[{$lang->getTag()}]" . $htmlTail);
			}, $this->getForm()->langsService->getLangs(false));
		}

		return $this->getForm()->getHttpData($type, $this->getHtmlName() . $htmlTail);
	}

	public function getAttrRules(): array
	{
		return Helpers::exportRules($this->getRules()) ?: [];
	}

	public function getAttrRulesJson(): string { return Json::encode($this->getAttrRules()); }
}
