<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Traits;

use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\SortableCheckboxList;
use Nette\Forms\Controls\ChoiceControl;
use Nette\Forms\Form;
use Nette\InvalidArgumentException;
use Nette\Utils\Html;


trait ChoiceInputTrait
{
	/**
	 * @var array items as user entered them - may be nested, unlike items, which are always flat.
	 */
	protected $rawItems;

	protected $rawValue;

	/**
	 * Processes an associative array in a way that it has no nesting. Keys for
	 * nested arrays are lost, but nested arrays are merged.
	 *
	 * @param array $array
	 *
	 * @return array
	 */
	public function flatAssocArray(array $array)
	{
		$ret = [];
		foreach ($array as $key => $value) {
			if (is_array($value)) {
				$ret += $this->flatAssocArray($value);
			} else {
				$ret[$key] = $value;
			}
		}

		return $ret;
	}

	/**
	 * Makes array of &lt;option&gt;. Can handle associative arrays just fine. Checks for duplicate values.
	 *
	 * @param array    $items
	 * @param callable $optionArgs     takes ($value,$caption) and spits out an array of &lt;option&gt;
	 *                                 attributes
	 * @param array    $valuesRendered for internal use. Do not change.
	 *
	 * @return array
	 * @throws InvalidArgumentException when $items have multiple of the same values
	 */
	public function makeOptionList($items, callable $optionArgs, array &$valuesRendered = [])
	{
		$ret = [];
		foreach ($items as $value => $caption) {
			if (is_int($value)) {
				$value = (string) $value;
			}

			if (is_array($caption)) {
				// subgroup
				$option = Html::el('optgroup', ['label' => $value]);

				// options within the group
				$nested = $this->makeOptionList($caption, $optionArgs, $valuesRendered);

				foreach ($nested as $item) {
					$option->addHtml($item);
				}
			} else {
				$caption = $this->translate($caption);
				if (in_array($value, $valuesRendered)) {
					throw new InvalidArgumentException("Value '$value' is used multiple times.");
				}
				$valuesRendered[] = $value;

				// normal option
				$option = Html::el('option',
					array_merge(['value' => (string) $value], $optionArgs($value, $caption)));
				$option->setText($caption);
			}
			$ret[] = $option;
		}

		return $ret;
	}

	/**
	 * @param array $items Items to set. Associative arrays are supported.
	 * @param  bool $useKeys
	 *
	 * @return static
	 */
	public function setItems(array $items, $useKeys = true)
	{
		/** @var ChoiceControl $this */
		$this->rawItems = $items;

		$processed = $this->flatAssocArray($items);
		/** @noinspection PhpUndefinedMethodInspection */
		parent::setItems($processed, $useKeys);

		return $this;
	}

	/**
	 * Check if whole control is disabled.
	 * This is false if only a set of values is disabled
	 * @return bool
	 */
	protected function isControlDisabled()
	{
		if (is_bool($this->disabled)) {
			return $this->disabled;
		}

		return false;
	}

	/**
	 * Check if a specific value is disabled. If whole control is disabled, returns false.
	 *
	 * @param $value mixed value to check for
	 *
	 * @return bool
	 */
	protected function isValueDisabled($value)
	{
		$disabled = $this->disabled;
		if (is_array($disabled)) {
			return isset($disabled[$value]) && $disabled[$value];
		} elseif (!is_bool($disabled)) {
			return $disabled == $value;
		}

		return false;
	}

	/**
	 * Self-explanatory
	 * @param $value
	 *
	 * @return bool
	 */
	protected function isValueSelected($value, $lang = null)
	{
		$val = $this->getValue();

		if (is_array($val) && $lang !== null) {
			return $value ? $val[$lang] == $value : false;
		} else if (is_null($value)) {
			return false;
		} elseif (is_array($val)) {
			return in_array($value, $val);
		}

		return $value == $val;
	}

	/**
	 * Sets control's value.
	 * @return static
	 * @internal
	 */
	public function setValue($value)
	{
		//		parent::setValue()
		//		if ($value === null) {
		//			$value = '';
		//		} elseif (!$this->getIsMultiLanguage() && !is_scalar($value) && !method_exists($value, '__toString')) {
		//			throw new Nette\InvalidArgumentException(sprintf("Value must be scalar or null, %s given in field '%s'.", gettype($value), $this->name));
		//		}
		$this->value    = $value;
		$this->rawValue = $this->getIsMultiLanguage() ? (array) $value : (string) $value;

		return $this;
	}

	public function loadHttpData()
	{
		$this->value = $this->getHttpData(Form::DATA_TEXT);

		$allowArray = in_array(get_called_class(), [CheckboxListInput::class, SortableCheckboxList::class]);

		if ($this->value !== null && !$allowArray) {
			if (is_array($this->disabled) && isset($this->disabled[$this->value])) {
				$this->value = null;
			} else if (!$this->getIsMultiLanguage() || !is_array($this->value)) {
				$this->value = key([$this->value => null]);
			}
		}
	}

	public function getValue()
	{
		if ($this->getIsMultiLanguage()) {
			foreach ($this->value as $lang => $val) {
				if (!array_key_exists($val, $this->items))
					$this->value[$lang] = null;
			}

			return $this->value;
		}

		return parent::getValue();
	}
}
