export default class NestedLists {
	constructor() {
		this.list = [];
		this.init();
	}

	init() {
		document.querySelectorAll('.nested-list:not(.init), .nested-table:not(.init)').forEach(el => {
			if (el.nestedInited === true)
				return;

			el.nestedInited = true;
			this.list.push(new NestedList(el));
		});
	};
}

class NestedList {
	constructor(wrap) {
		const self         = this;
		this.nestedListKey = wrap.dataset.nestedListKey;
		this.wrap          = wrap;
		this.storage       = window.localStorage;
		this.storageName   = wrap.dataset.storageName;

		self.openChecked();
		self.initButtons();

		// Kvůli kompatibilitě s jquery
		window.jQuery(document).ajaxComplete(function(event, xhr, settings) {
			self.openChecked();
			self.initButtons();
		});

		global.naja.addEventListener('complete', e => {
			self.openChecked();
			self.initButtons();
		});

		this.wrap.addEventListener('click', e => {
			if (e.target.matches('.fa-minus-square, .fa-plus-square')) {
				self.arrowClick(e.target.parentElement);
			} else if (e.target.matches('.state--radio-button') && e.target.previousElementSibling.checked) {
				const target = e.target.previousElementSibling;

				target.removeAttribute('checked');
				target.checked = false;
				target.dispatchEvent(new Event('change'));
			}
		});

		this.wrap.querySelectorAll('.nested-table__head [data-search-key]').forEach(input => {
			let timeout = null;
			input.addEventListener('keyup', e => {
				if (timeout) {
					clearTimeout(timeout);
				}

				timeout = setTimeout(() => {
					let searchKey = input.dataset.searchKey;
					let searchVal = input.value.toLowerCase();

					this.wrap.querySelectorAll('.nested-search-result-contain').forEach(el => {
						el.classList.remove('nested-search-result-contain');
					});

					this.wrap.querySelectorAll('.search-result-contain').forEach(el => {
						el.classList.remove('search-result-contain');
					});

					if (searchVal.length > 1) {
						this.wrap.querySelectorAll('.nested-table__item [data-search-key="' + searchKey + '"]').forEach(item => {
							let itemValue = item.dataset.searchValue?.toLowerCase();
							if (itemValue.includes(searchVal)) {
								let parent = item.closest('.nested-table__head');

								if (parent) {
									parent.classList.add('search-result-contain');

									let i          = 0;
									let parentItem = item.closest('.nested-table__item');
									while (parentItem && i <= 100) {
										let head = parentItem.querySelector('.nested-table__head');
										if (head) {
											head.classList.add('nested-search-result-contain');
										}

										parentItem = parentItem.parentElement.closest('.nested-table__item');

										i++;
									}
								}
							}
						});
					}
				}, 300);
			});
		});
	}

	openChecked() {
		let saved = this.getStorageData();

		if (saved)
			saved.forEach(id => {
				let item = this.wrap.querySelector('[data-id="' + id + '"]');

				if (item)
					item.classList.add('active');
			});

		this.wrap.querySelectorAll('input:checked').forEach(input => {
			let item = input.parentNode.parentNode.parentNode;
			let i    = 0;

			while (item && !item.classList.contains('active') && (item.classList.contains('nested-list__item') || item.classList.contains('nested-table__item')) && i < 20) {
				item.classList.add('active');

				item = item.parentNode.parentNode;
				i++;
			}
		});
	}

	arrowClick(arrow) {
		let item = arrow.parentNode.parentNode;

		if (item.classList.contains('active')) {
			item.classList.remove('active');

			if (item.dataset.id) {
				this.removeFromStorage(item.dataset.id);

				item.querySelectorAll('[data-id]').forEach(el => {
					this.removeFromStorage(el.dataset.id);
					el.classList.remove('active');
				});
			}
		} else {
			item.classList.add('active');

			if (item.dataset.id) {
				this.addToStorage(item.dataset.id);
			}
		}
	}

	getStorageData() {
		if (!this.storageName)
			return null;

		let data = this.storage.getItem(this.storageName);

		return data ? JSON.parse(data) : [];
	}

	addToStorage(id) {
		let data = this.getStorageData();

		if (data === null)
			return;

		data.push(id);
		data = [...new Set(data)];

		this.saveToStorage(data);
	}

	removeFromStorage(id) {
		let data = this.getStorageData();

		if (!data)
			return;

		let index = data.indexOf(id);
		if (index > -1) {
			data.splice(index, 1);
		}

		this.saveToStorage(data);
	}

	saveToStorage(data) {
		this.storage.setItem(this.storageName, JSON.stringify(data));
	}

	initButtons() {
		if (!this.nestedListKey) {
			return;
		}

		const self             = this;
		const removeCheckedBtn = document.querySelector('[data-nested-list-remove-checked="' + self.nestedListKey + '"]');
		if (removeCheckedBtn && !removeCheckedBtn.nestedListInited) {
			removeCheckedBtn.addEventListener('click', e => {
				removeCheckedBtn.nestedListInited = true;
				e.preventDefault();

				self.wrap.querySelectorAll('input:checked, input:checked').forEach(input => {
					input.checked = false;
				});

				return false;
			});
		}
	}
}
