/**
 * Třída pro tvorbu map využívající funkcionalitu LeafletJS a mapových podkladů Seznam.cz.
 * Nepoužívá zastaralé JS SDK, nýbrž REST API vydané v létě 2023.
 */

const isMobile = {
	Android: function () {
		return navigator.userAgent.match(/Android/i);
	},
	BlackBerry: function () {
		return navigator.userAgent.match(/BlackBerry/i);
	},
	iOS: function () {
		return navigator.userAgent.match(/iPhone|iPad|iPod/i);
	},
	Opera: function () {
		return navigator.userAgent.match(/Opera Mini/i);
	},
	Windows: function () {
		return navigator.userAgent.match(/IEMobile/i);
	},
	any: function () {
		return (isMobile.Android() || isMobile.BlackBerry() || isMobile.iOS() || isMobile.Opera() || isMobile.Windows());
	}
}

export class SeznamMap {
	/**
	 * @param {Object} options enableSwitchLayer - prepinani vrstev mapy, scrollWheelZoom - povolení zoomani kolečkem, dragging - povoleni dragging mouse/touch, tap - povoleni tapovani, texts - textace
	 */
	constructor(apiKey, elementId, callback, initImmediately = true, options = {}) {
		this.apiKey = apiKey
		this.elementId = elementId
		this.callback = callback

		if (!options.hasOwnProperty('scrollWheelZoom')) {
			options.scrollWheelZoom = false
		}

		if (!options.hasOwnProperty('dragging')) {
			options.dragging = !isMobile.any()
		}

		if (!options.hasOwnProperty('tap')) {
			options.tap = !isMobile.any()
		}

		if (options.hasOwnProperty('clustering')) {
		}

		this.options = options

		if (initImmediately) {
			this.init();
		}
	}

	init() {
		const initMap = () => {
			this.map = new L.Map(this.elementId, this.options)

			const basicLayer = L.tileLayer(`https://api.mapy.cz/v1/maptiles/basic/256/{z}/{x}/{y}?apikey=${this.apiKey}`, {
				minZoom: 0,
				maxZoom: 19,
				attribution: '<a href="https://api.mapy.cz/copyright" target="_blank">&copy; Seznam.cz a.s. a další</a>',
				detectRetina: true
			})
			const outdoorLayer = L.tileLayer(`https://api.mapy.cz/v1/maptiles/outdoor/256/{z}/{x}/{y}?apikey=${this.apiKey}`, {
				minZoom: 0,
				maxZoom: 19,
				attribution: '<a href="https://api.mapy.cz/copyright" target="_blank">&copy; Seznam.cz a.s. a další</a>',
				detectRetina: true
			})
			const aerialLayer = L.tileLayer(`https://api.mapy.cz/v1/maptiles/aerial/256/{z}/{x}/{y}?apikey=${this.apiKey}`, {
				minZoom: 0,
				maxZoom: 19,
				attribution: '<a href="https://api.mapy.cz/copyright" target="_blank">&copy; Seznam.cz a.s. a další</a>',
				detectRetina: true
			})

			let tileLayers = []
			tileLayers[this.options.texts?.layerBasicCaption ?? 'Základní'] = basicLayer
			tileLayers[this.options.texts?.layerOutdoorCaption ?? 'Turistická'] = outdoorLayer
			tileLayers[this.options.texts?.layerAerialCaption ?? 'Letecká'] = aerialLayer

			// vychozi vrstva
			basicLayer.addTo(this.map)

			// moznost menit vrstvy
			if (this.options['enableSwitchLayer']) {
				L.control.layers(tileLayers).addTo(this.map)
			}

			const LogoControl = L.Control.extend({
				options: {
					position: 'bottomleft',
				},

				onAdd: function (map) {
					const container = L.DomUtil.create('div')
					const link = L.DomUtil.create('a', '', container)

					link.setAttribute('href', 'https://mapy.cz/')
					link.setAttribute('target', '_blank')
					link.innerHTML = '<img src="https://api.mapy.cz/img/api/logo.svg" />'
					L.DomEvent.disableClickPropagation(link)

					return container
				},
			})

			new LogoControl().addTo(this.map)

			this.callback(this)
		}

		const scriptId = 'smap-' + this.elementId
		if (document.getElementById(scriptId)) {
			initMap()
			return
		}

		const stylesheet = document.createElement('link')
		stylesheet.rel = 'stylesheet'
		stylesheet.href = 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css'
		stylesheet.integrity = 'sha256-p4NxAoJBhIIN+hmNHrzRCf9tD/miZyoHS5obTRR9BMY='
		stylesheet.crossOrigin = ''
		document.head.append(stylesheet)

		const script = document.createElement('script')
		script.src = 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js'
		script.integrity = 'sha256-20nQCchB9co0qIjJZRGuk2/Z9VM+kNiyxNV1lvTlZBo='
		script.id = scriptId
		script.crossOrigin = ''

		if (!this.options.clustering) {
			script.addEventListener('load', initMap)
		} else {
			script.addEventListener('load', () => {
				const markerClusterJs = document.createElement('script')
				markerClusterJs.src = 'https://unpkg.com/leaflet.markercluster/dist/leaflet.markercluster.js'
				markerClusterJs.addEventListener('load', initMap)
				document.head.append(markerClusterJs)
			})
		}
		document.head.append(script)

		if (this.options.clustering) {
			const markerClusterCss = document.createElement('link');
			markerClusterCss.rel = 'stylesheet'
			markerClusterCss.href = 'https://unpkg.com/leaflet.markercluster/dist/MarkerCluster.css'
			document.head.append(markerClusterCss)
			const markerClusterDefaultCss = document.createElement('link')
			markerClusterDefaultCss.rel = 'stylesheet'
			markerClusterDefaultCss.href = 'https://unpkg.com/leaflet.markercluster/dist/MarkerCluster.Default.css'
			document.head.append(markerClusterDefaultCss)
		}
	}

	/**
	 * Nastavi zamereni skupinu bodů
	 *
	 * @param {Array} marker (Marker[])
	 * @return Array
	 */
	addMarkers(markers) {
		let clusterMarkers = null
		if (this.options.clustering) {
			clusterMarkers = L.markerClusterGroup()
		}

		const lMarkers = []
		markers.forEach(m => {
			let lMarker = null
			if (this.options.clustering) {
				lMarker = this.createLMarker(m)
				clusterMarkers.addLayer(lMarker)
			} else {
				lMarker = this.addMarker(m)
			}

			if (lMarker) {
				lMarkers.push(lMarker)
			}
		})

		if (this.options.clustering) {
			this.map.addLayer(clusterMarkers)
		}

		return lMarkers
	}

	/**
	 * Nastavi zamereni na 1 bod
	 *
	 * @param {Coord} coord
	 * @param {Number} zoom
	 */
	setView(coord, zoom) {
		this.map.setView(coord.toArray(), zoom)
	}

	/**
	 * Nastavi zamereni skupinu bodů
	 *
	 * @param {Array} marker (Marker[])
	 * @param {Number} zoom
	 */
	setBoundingView(markers, zoom = null) {
		let ms = []
		markers.forEach(el => {
			ms.push(el.toLMarker())
		})

		let group = new L.featureGroup(ms)

		this.map.fitBounds(group.getBounds())

		if (zoom) {
			this.map.setZoom(zoom)
		}
	}

	/**
	 * @param {Marker} marker
	 * @return {Object}
	 */
	addMarker(marker) {
		return this.createLMarker(marker).addTo(this.map)
	}

	/**
	 * @param {Marker} marker
	 * @return {L.Marker}
	 */
	createLMarker(marker) {
		const m = marker.toLMarker()

		if (marker.popupContent) {
			m.bindPopup(marker.popupContent, {className: 'smap-popup'})
		}

		return m
	}

}

export class Coord {
	constructor(lat, lng) {
		this.lat = lat
		this.lng = lng
	}

	toArray() {
		return [this.lat, this.lng]
	}
}

export class Marker {
	/**
	 * @param {Coord} coord
	 * @param {Icon} icon
	 * @param {String|null} popupContent
	 */
	constructor(coord, icon, popupContent = null, extras = {}) {
		this.coord = coord
		this.icon = icon
		this.popupContent = popupContent
		this.extras = extras
	}

	toLMarker() {
		const marker = L.marker(this.coord.toArray(), this.icon ? {
			icon: L.icon({
				iconUrl: this.icon.url,
				iconSize: this.icon.size
			})
		} : null)
		marker.extras = this.extras
		return marker
	}
}

export class Icon {
	/**
	 * @param {String} url
	 * @param {Array} size
	 */
	constructor(url, size) {
		this.url = url
		this.size = size
	}
}

/**
 * @param {SeznamMap} map
 * @param {Array} coords [{lat: 20.222, lng: 40.4005}]
 * @param {string} markerImgSrc
 * @param {Array} markerSize
 * @param {Number|null} zoom
 */
export function fastMapConfiguration(map, coords = [], markerImgSrc, markerSize, zoom = null) {
	const markers = []
	let clusterMarkers = null
	if (map.options.clustering) {
		clusterMarkers = L.markerClusterGroup()
	}
	coords.forEach(el => {
		const m = new Marker(new Coord(el.lat, el.lng), markerImgSrc ? new Icon(markerImgSrc, markerSize) : null)
		if (map.options.clustering) {
			clusterMarkers.addLayer(m.toLMarker())
		} else {
			map.addMarker(m)
		}
		markers.push(m)
	})

	if (map.options.clustering) {
		map.map.addLayer(clusterMarkers)
	}
	map.setBoundingView(markers, zoom)
}
