<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Country;

use Exception;
use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\Entities\CountryText;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\TextInput;
use Doctrine\Common\Collections\ArrayCollection;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class CountryForm extends BaseControl
{
	protected ?Country $country = null;

	public function __construct(
		protected Countries $countriesService
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'default.name2')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addText('id', 'admin.countryForm.idCountry')
			->setMaxLength(2);
		$form->addText('iso3166_1', 'admin.countryForm.iso3166_1')
			->setMaxLength(3)
			->setRequired();
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			/** @var CountryText[] $texts */
			$texts = [];

			if ($this->country) {
				$country = $this->country;
				$texts   = $country->getTexts()->toArray();
			} else {
				$country = new Country($values->id, $values->iso3166_1);
			}

			$langValues = $form->convertMultilangValuesToArray();
			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['name']) {
					$texts[$l] = new CountryText($country, $l);
				}

				if ($v['name'] === '' || $texts[$l]->name === '') {
					if (!Validators::isNone($texts[$l])) {
						$this->em->remove($texts[$l]);
					}

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->name = $v['name'];

				$this->em->persist($texts[$l]);
			}

			$country->texts = new ArrayCollection($texts);
			$country->setIso3166_1($values->iso3166_1);

			$this->em->persist($country);
$this->em->flush();

			$form->addCustomData('countryId', $country->getId());

			$this->presenter->flashMessageSuccess('default.saved');
			$this->presenter->redrawControl('flashes');

			$this->em->commit();

			$cache = new Cache($this->cacheStorage, Countries::CACHE_NAMESPACE);
			foreach ($this->langsService->getLangs(false) as $lang) {
				$cache->remove(Countries::CACHE_NAMESPACE . '/' . $lang->getTag());
			}
		} catch (Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
		$this->redrawControl('form');

		return false;
	}

	public function setCountry(string $id): void
	{
		$this->country = $this->countriesService->get($id);

		if ($this->country) {
			$o = $this->country;

			/** @var TextInput $id */
			$id = $this['form']->getComponent('id');
			$id->setDisabled(true);

			$d = [
				'id'        => $o->getId(),
				'iso3166_1' => $o->getIso3166_1(),
			];

			foreach ($o->getTexts() as $l => $text) {
				$d['name'][$l] = $text->name;
			}

			$this['form']->setDefaults($d);
		} else {
			$this->presenter->error();
		}
	}
}
