<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Country;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use Nette\Utils\Html;

class CountryGrid extends BaseControl
{
	public function __construct(
		protected Countries $countriesService
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->countriesService->getEr()
			->createQueryBuilder('c')
			->innerJoin('c.texts', 't', Join::WITH, 't.lang = :lang')
			->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
			->orderBy('c.position', 'ASC');

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('countryGrid:gridSortableRow!');

		//Columns
		$grid->addColumnText('id', 'admin.countryForm.idCountryShort');
		$grid->addColumnText('name', 'default.name')->setRenderer(function(Country $country) {
			return Html::el('a', [
				'href' => $this->presenter->link('Country:editCountry', $country->getId()),
			])->setText($country->getText()->name);
		});

		// Actions
		$grid->addAction('edit', '', 'Country:editCountry')
			->setIcon('edit')
			->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setConfirm('default.reallyDelete')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax');

		$grid->getColumn('id')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(string $id): void
	{
		if ($this->countriesService->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function handleGridSortableRow(): void
	{
		$request  = $this->presenter->getHttpRequest();
		$id       = $request->getPost('id');
		$position = $request->getPost('position');

		if ($id !== null && $position !== null && $this->countriesService->setPosition((string) $id, (int) $position)) {
			$this->presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$this->presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this->presenter->redrawControl('flashes');
		$this->presenter->redrawControl('countryGrid');
	}

}
