<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\PublicHoliday;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\AdminModule\Model\PublicHolidays;
use Core\Model\Countries;
use Core\Model\Entities\PublicHoliday;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use Nette\Utils\Html;

class PublicHolidaysGrid extends BaseControl
{
	protected Countries                 $countriesService;
	protected PublicHolidays            $publicHolidays;
	protected IPublicHolidayFormFactory $formFactory;

	public function __construct(
		Countries                 $countries,
		PublicHolidays            $publicHolidays,
		IPublicHolidayFormFactory $formFactory
	)
	{
		$this->countriesService = $countries;
		$this->publicHolidays   = $publicHolidays;
		$this->formFactory      = $formFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('admin.title.newPublicHoliday');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setPublicHoliday($id);
		$this->template->modalTitle = $this->t('admin.title.editPublicHoliday',
			['title' => $this['form']->publicHoliday->date->format('j. n. Y')]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDownload(): void
	{
		if ($this->publicHolidays->downloadFromApi()) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function handleDelete(string $id): void
	{
		if ($this->publicHolidays->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->publicHolidays->getEr()
			->createQueryBuilder('h')
			->addSelect('ht')
			->innerJoin('h.texts', 'ht', Join::WITH, 'ht.lang = :lang')
			->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
			->orderBy('h.date', 'DESC');

		$grid->setDataSource($qb);

		//Columns
		$grid->addColumnDateTime('date', 'admin.publicHoliday.date')
			->setFormat('j. n. Y');
		$grid->addColumnText('title', 'admin.publicHoliday.title')->setRenderer(function(PublicHoliday $row): Html {
			return Html::el()->setHtml(nl2br((string) $row->getText()->title));
		});

		// Actions
		$grid->addAction('edit', '', 'edit!')
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax')
			->setRenderCondition(function(PublicHoliday $row): bool {
				return $row->source !== 'apiPshk';
			});
		$grid->addAction('delete', '', 'delete!')
			->setConfirm('default.reallyDelete')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setRenderCondition(function(PublicHoliday $row): bool {
				return $row->source !== 'apiPshk';
			});

		$grid->addToolbarButton('download!', 'admin.publicHoliday.download')
			->setIcon('download')
			->setClass('ajax btn-success btn');
		$grid->addToolbarButton('add!', 'default.add')
			->setIcon('plus')
			->setClass('ajax btn-success btn');

		$grid->getColumn('date')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentForm(): PublicHolidayForm
	{
		$control = $this->formFactory->create();
		if ($this->getParameter('id')) {
			$control->setPublicHoliday((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->presenter->redrawControl('flashes');
			$this->handleEdit($form->getCustomData('publicHolidayId'));
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}
}
