<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Redirects;

use Core\AdminModule\Model\Redirects;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;

class RedirectsGrid extends BaseControl
{
	public function __construct(protected Redirects $redirectsService, protected IRedirectFormFactory $formFactory)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('core.title.addRedirect');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('core.title.editRedirect');
		$this['form']->setRedirect($id);
		$this->template->modal = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$this->removeRedirects([$id]);
	}

	public function removeRedirects(array $ids): void
	{
		if ($this->redirectsService->removeRedirects($ids)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->redirectsService->getEr()->createQueryBuilder('r')
			->orderBy('r.id', 'desc');

		$grid->setDataSource($qb);

		// Column
		$grid->addColumnText('title', 'core.redirects.title')
			->setFilterText();
		$grid->addColumnText('from', 'core.redirects.from')
			->setFilterText();
		$grid->addColumnText('to', 'core.redirects.to')
			->setFilterText();
		$grid->addColumnText('siteIdent', 'core.redirects.siteIdent')
			->setFilterText();
		$grid->addColumnText('lang', 'core.redirects.lang')
			->setFilterText();
		$grid->addColumnText('redirectCode', 'core.redirects.redirectCode')
			->setFilterText();
		$grid->addColumnText('package', 'core.redirects.package')
			->setFilterText();
		$grid->addColumnText('relationKey', 'core.redirects.relationKey')
			->setFilterText();
		$grid->addColumnDateTime('created', 'core.redirects.created')
			->setFormat('j. n. Y H:i:s');

		// Prototype
		$grid->getColumn('redirectCode')->getElementPrototype('td')->class[] = 'w1nw';

		// Actions
		$grid->addAction('edit', '', 'edit!')
			->setIcon('fas fa-pencil-alt')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->addClass('ajax')
			->setBsType('danger')
			->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.delete')->onSelect[] = $this->removeRedirects(...);

		$grid->getColumn('created')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentForm(): RedirectForm
	{
		$control = $this->formFactory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}
}
