<?php declare(strict_types = 1);

namespace Core\AdminModule\Components;

use Core\Model\Entities\StaticTextTranslation;
use Core\Model\StaticTexts;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\UI\Multiplier;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;
use Tracy\Debugger;
use function str_starts_with;

class StaticTextsForm extends BaseControl
{
	protected array $messages = [];

	public function __construct(
		protected IStaticTextsFormFactory $staticTextsFormFactory,
		protected StaticTexts             $staticTexts,
	)
	{
	}

	public function render(): void
	{
		$this->refresh();
		$this->template->messages = $this->staticTexts->getMessagesWithResources();
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param IComponent $presenter
	 *
	 * @throws Exception
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$this->refresh();
	}

	public function createComponentStaticTextForm(): Multiplier
	{
		return new Multiplier(fn($domain): Multiplier => new Multiplier(function($locale) use ($domain): BaseForm {
			$loc = 'cs_CZ';
			foreach ($this->messages[$domain] as $key => $v) {
				if (str_starts_with($key, $this->translator->getLocale())) {
					$loc = $key;
					break;
				}
			}
			$messages              = $this->messages[$domain][$locale];
			$currentLocaleMessages = $this->messages[$domain][$loc];

			$form = $this->createForm();
			$form->setAjax();
			foreach ($messages as $key => $message) {
				$form->addText(
					str_replace('.', '_', $key),
					$currentLocaleMessages[$key]['db'] ?? $currentLocaleMessages[$key]['file'],
				)
					->setTranslator(null)
					->setPlaceholder($message['db'] ?? $message['file'])
					->setDefaultValue($message['db']);
			}

			$form->addCustomData('locale', $locale);
			$form->addCustomData('domain', $domain);
			$form->addSubmit('submit', $this->translator->translate('default.save') . ' ' . $locale)
				->setFloating();

			$form->onSuccess[] = $this->onSuccess(...);

			return $form;
		}));
	}

	public function onSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$this->em->beginTransaction();

			foreach ($values as $key => $value) {
				$key        = str_replace('_', '.', $key);
				$locale     = $form->getCustomData('locale');
				$staticText = $this->staticTexts->getOneByKeyLocale($key, $locale);

				if ($staticText && Validators::isNone($value)) { // remove
					$this->em->remove($staticText);
				} else if ($staticText && !Validators::isNone($value)) { // edit
					$staticText->message = $value;
					$this->em->persist($staticText);
				} else if (!Validators::isNone($value)) { // add
					$this->em->persist(new StaticTextTranslation($key, $locale, $value));
				}
			}

			$this->em->flush();
			$this->em->commit();
			$this->staticTexts->clearCache();

			if (($presenter = $this->presenter) !== null) {
				$presenter->flashMessageSuccess('default.saved');
				$presenter->redrawControl('flashes');
				$this->redrawControl('form');
			}
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();

			if (($presenter = $this->presenter) !== null) {
				$presenter->flashMessageDanger('defeault.removeFailed');
			}
		}
	}

	public function refresh(): void
	{
		$this->staticTexts->merge();
		$this->messages = $this->staticTexts->getMessagesWithResources();
	}

}
