<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Variables;

use Core\AdminModule\Model\Variables;
use Core\Model\Helpers\VariableHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;

class VariablesGrid extends BaseControl
{
	protected Variables            $variables;
	protected IVariableFormFactory $variableFormFactory;
	protected VariableHelper       $variableHelper;

	public function __construct(
		Variables            $variables,
		IVariableFormFactory $variableFormFactory,
		VariableHelper       $variableHelper
	)
	{
		$this->variables           = $variables;
		$this->variableFormFactory = $variableFormFactory;
		$this->variableHelper      = $variableHelper;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$data = [];
		foreach ($this->variables->getEr()->createQueryBuilder('v')
			         ->groupBy('v.key')
			         ->getQuery()->getArrayResult() as $row) {
			$data[$row['key']] = [
				'id'    => $row['key'],
				'value' => $row['value'],
			];
		}

		$grid->setDataSource($data);

		$grid->addColumnText('id', 'core.variable.key');
		$grid->addColumnText('value', 'core.variable.value')->setRenderer(function(array $row) {
			$formatted = $this->variableHelper->formatValue($row['value']);

			if ($formatted !== $row['value']) {
				return $row['value'] . ' (' . $formatted . ')';
			}

			return $row['value'];
		});

		$grid->addAction('edit', '', 'edit!')
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	public function createComponentForm(): VariableForm
	{
		$control = $this->variableFormFactory->create();

		if ($this->getParameter('id')) {
			$control->id = (string) $this->getParameter('id');
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this['grid']->reload();
			$this->handleEdit($form->getCustomData('key'));
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this['grid']->reload();
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('core.title.addVariable');
		$this->template->modal      = 'variableForm';
		$this->redrawControl('modal');
	}

	public function handleEdit(string $id): void
	{
		$this['form']->setVariable($id);
		$this->template->modalTitle = $this->t('core.title.editVariable', ['key' => $id]);
		$this->template->modal      = 'variableForm';
		$this->redrawControl('modal');
	}

	public function handleDelete(string $id): void
	{
		if ($this->variables->removeByKey($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

}
