<?php declare(strict_types = 1);

namespace Core\AdminModule\Model;

use DateTime;
use Exception;
use Core\Model\Countries;
use Core\Model\Entities\PublicHoliday;
use Core\Model\Entities\PublicHolidayText;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Lang\Langs;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Nette\Caching\Cache;
use Nette\Utils\Json;
use Tracy\Debugger;

/**
 * @method PublicHoliday|null getReference($id)
 * @method PublicHoliday[]|null getAll()
 * @method PublicHoliday|null get($id)
 */
class PublicHolidays extends BaseEntityService
{
	protected $entityClass = PublicHoliday::class;

	protected Countries $countries;
	protected Langs     $langs;

	public function __construct(
		Countries $countries,
		Langs     $langs
	)
	{
		$this->countries = $countries;
		$this->langs     = $langs;
	}

	public function downloadFromApi(): bool
	{
		try {
			$now = (new DateTime())->format('Y-m-d');
			$data = Json::decode((string) file_get_contents('https://api-evidence.pshk.cz/v1/cms/statni-svatky'), Json::FORCE_ARRAY);

			if ($data['status'] === 'success') {
				$siteData = [];
				foreach ($this->getEr()->createQueryBuilder('ph')
					         ->select('ph.id, ph.date')
					         ->where('ph.source = :source')
					         ->setParameter('source', 'apiPshk')
					         ->getQuery()->getScalarResult() as $row) {
					$siteData[$row['date']] = $row['id'];
				}

				$apiData = [];
				foreach ($data['data'] as $year => $rows) {
					foreach ($rows as $date => $title) {
						if ($date < $now) {
							continue;
						}

						$apiData[$date] = $title;
					}
				}

				$czCountry = $this->countries->get('CZ');

				// new
				foreach (array_diff_key($apiData, $siteData) as $date => $title) {
					$date = DateTime::createFromFormat('Y-m-d', $date);

					if ($date) {
						$ph         = new PublicHoliday($date);
						$ph->source = 'apiPshk';

						if ($czCountry) {
							$ph->country = $czCountry;
						}

						$phText        = new PublicHolidayText($ph, 'cs');
						$phText->title = $title;

						$ph->texts->set('cs', $phText);

						$this->em->persist($ph);
						$this->em->persist($phText);
					}
				}

				//old
				$toRemove = array_diff_key($siteData, $apiData);
				if (!empty($toRemove)) {
					$this->em->createQuery('DELETE FROM ' . PublicHoliday::class . ' ph WHERE ph.id IN (:ids)')
						->setParameter('ids', array_values($toRemove))
						->execute();
				}

				$this->em->flush();

				$this->em->clear();
			} else {
				throw new Exception('PublicHolidays downloadFromApi failed - ' . $data['message']);
			}

			$this->clearCache();

			return true;
		} catch (Exception $e) {
			Debugger::log($e, Debugger::WARNING);
			LogNotifier::toDevelopers($e->getMessage(), 'PublicHolidays downloadFromApi failed');
		}

		return false;
	}

	public function afterRemove(): void
	{
		$this->clearCache();
	}

	public function clearCache(): void
	{
		$cache = new Cache($this->cacheStorage, 'corePublicHolidays');

		foreach ($this->langs->getLangs(false) as $lang) {
			$cache->remove('all/' . $lang->getTag());
		}
	}
}
