<?php declare(strict_types = 1);

namespace Core\AdminModule\Presenters;

use Exception;
use Core\Model\Translation\FieldLockerService;
use Core\Model\Translation\TextTranslator;
use DeepL\QuotaExceededException;
use Nette\Utils\Json;
use Nette\Utils\Validators;

class TranslatorPresenter extends BasePresenter
{
	public function __construct(
		protected TextTranslator     $textTranslator,
		protected FieldLockerService $fieldLocker,
	)
	{
	}

	public function actionDeepl(): void
	{
		try {
			$key        = (string) $this->getHttpRequest()->getPost('key');
			$text       = $this->getHttpRequest()->getPost('text');
			$sourceLang = $this->getHttpRequest()->getPost('sourceLang');
			$targetLang = $this->getHttpRequest()->getPost('targetLang');

			if ($this->fieldLocker->getLocked($key, $targetLang)) {
				$this->flashMessageDanger('admin.translateLocked');
			} else if (!$text || !$sourceLang || !$targetLang) {
				$this->flashMessageDanger('default.error');
			} else {
				$this->payload->result = $this->textTranslator->translateHtml($text, $sourceLang, $targetLang, true);

				$this->flashMessageSuccess('admin.translator.translated');
			}
		} catch (QuotaExceededException $e) {
			$this->flashMessageDanger('admin.translator.quotaExceeded');
		} catch (Exception $e) {
			$this->flashMessageDanger('default.error');
		}

		$this->redrawControl('flashes');
	}

	public function actionDeeplMulti(): void
	{
		try {
			$sourceLang = (string) $this->getHttpRequest()->getPost('sourceLang');
			$data       = Json::decode((string) $this->getHttpRequest()->getPost('data'), Json::FORCE_ARRAY);

			$resultData = [];

			foreach ($data as $targetLang => $fields) {
				$filteredText = [];
				$filteredHtml = [];

				foreach ($fields as $field) {
					$tmp = explode('-', $field['key']);
					if (strlen(end($tmp)) !== 2) {
						continue;
					}

					if (!isset($this->fieldLocker->getLockedFields()[$field['key']])) {
						if (Validators::isUrl($field['value']) || Validators::isUri($field['value'])) {
							$resultData[$targetLang][] = [
								'key'    => $field['key'],
								'result' => $field['value'],
							];

							continue;
						}

						if ($field['value'] === '') {
							continue;
						}

						if (strip_tags($field['value']) === $field['value']) {
							$filteredText[] = $field;
						} else {
							$filteredHtml[] = $field;
						}
					}
				}

				$data = array_map(static fn($v) => $v['value'], $filteredHtml);

				if (!empty($data)) {
					foreach ($this->textTranslator->translateHtmlMulti($data, $sourceLang, $targetLang) as $k => $text) {
						$resultData[$targetLang][] = [
							'key'    => $filteredHtml[$k]['key'],
							'result' => $text,
						];
					}
				}

				$data = array_map(static fn($v) => $v['value'], $filteredText);

				if (!empty($data)) {
					foreach ($this->textTranslator->translateTextMulti($data, $sourceLang, $targetLang) as $k => $text) {
						$resultData[$targetLang][] = [
							'key'    => $filteredText[$k]['key'],
							'result' => $text,
						];
					}
				}
			}

			$this->flashMessageSuccess('admin.translator.translated');

			$this->payload->result = $resultData;
		} catch (QuotaExceededException $e) {
			$this->flashMessageDanger('admin.translator.quotaExceeded');
		} catch (Exception $e) {
			$this->flashMessageDanger('default.error');
		}

		$this->redrawControl('flashes');
	}

	public function actionLock(): void
	{
		try {
			$key    = (string) $this->getHttpRequest()->getPost('key');
			$locale = (string) $this->getHttpRequest()->getPost('locale');
			$value  = $this->getHttpRequest()->getPost('value') === 'true';

			if (!$key || !$locale) {
				$this->flashMessageDanger('default.error');
			} else {
				$this->fieldLocker->saveData($key, $locale, $value);
				$this->flashMessageSuccess('admin.translator.' . ($value ? 'locked' : 'unlocked'));

				$this->payload->result = $value;
			}
		} catch (Exception $e) {
			$this->flashMessageDanger('default.error');
		}

		$this->redrawControl('flashes');
	}
}
