<?php declare(strict_types = 1);

namespace Core\Components\Header;

use Core\FrontModule\Model\SeoParser;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;

class Meta extends BaseControl
{
	protected array $metas            = [];
	protected array $otherHeaders     = [];
	public bool     $disableCanonical = false;

	public function __construct(
		protected SeoParser $seoParserService,
		protected Sites     $sitesService,
	)
	{
	}

	public function render(): void
	{
		if ($this->sitesService->getCurrentSite()->getCurrentDomain()->disableIndex) {
			$this->setMeta('robots', 'noindex, nofollow');
		}

		if (!$this->getMeta('og:description')) {
			$this->setMeta('og:description', $this->getMeta('description')['value'] ?? '', 'property');
		}

		$metas = $this->getMeta();

		if (!isset($metas['description']) && isset($metas['og:description'])) {
			$value = str_replace(["\n", "\r"], ' ', strip_tags((string) $metas['og:description']['value']));
			$value = str_replace('&nbsp;', ' ', $value);
			$value = preg_replace('/\s+/', ' ', $value);

			$metas['description'] = [
				'key'       => 'description',
				'value'     => $value,
				'attribute' => 'name',
			];
		}

		if (isset($metas['description']) && SystemConfig::load('metaDescriptionMaxLength')) {
			$metas['description']['value'] = Strings::truncate((string) $metas['description']['value'], (int) SystemConfig::load('metaDescriptionMaxLength'), '');

			$this->setMeta('og:description', $metas['description']['value'], 'property');
		}

		if (!$this->disableCanonical) {
			if (isset($metas['canonical'])) {
				$canonical = $metas['canonical']['value'] === 'self'
					? $this->presenter->link('//this')
					: $metas['canonical']['value'];
				unset($metas['canonical']);
			} else if ($this->presenter) {
				$canonicalParams = [];

				foreach ($this->presenter->getComponents() as $component) {
					if (method_exists($component, 'getPaginator')) {
						$page = $component->getPaginator()->getPage();

						if ($page && $page > 1) {
							$canonicalParams[$component->getName() . '-page'] = $page;
						}

						break;
					}
				}

				$canonical = $this->presenter->link('//this', $canonicalParams);
			} else {
				$canonical = $this->presenter->link('//this');
			}

			$this->template->canonical = $canonical;
		}

		$this->template->metas        = $metas;
		$this->template->otherHeaders = $this->otherHeaders;

		$this->template->render(__DIR__ . '/Meta.latte');
	}

	/*******************************************************************************************************************
	 * ======================== GET
	 */
	public function getMeta(?string $key = null): array|string
	{
		$metas = $this->metas;

		if ($key) {
			$metas = $metas[$key] ?? [];
		}

		return $key && isset($metas[$key]) ? $metas[$key] : $metas;
	}

	/*******************************************************************************************************************
	 * ======================== SET
	 */

	public function setMeta(string $key, ?string $value, string $attribute = 'name'): self
	{
		$this->metas[$key] = [
			'key'       => $key,
			'value'     => $this->seoParserService->parse((string) $value),
			'attribute' => $attribute,
		];

		return $this;
	}

	public function setOtherHeader(string $string): self
	{
		$this->otherHeaders[] = $string;

		return $this;
	}
}
