<?php declare(strict_types = 1);

namespace Core\Console\Content;

use Core\AdminModule\Model\EditorDataService;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Parameters;
use Core\Model\Sites;
use Doctrine;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'core:content:check-missing-editor-data')]
class CheckMissingEditorData extends Command
{
	protected array $aclData = [];

	public function __construct(
		protected EditorDataService  $editorDataService,
		protected MailBuilderFactory $mailBuilderFactory,
		protected Sites              $sites,
	)
	{
		parent::__construct();
	}

	protected function configure(): void
	{
		$this->setName('core:content:check-missing-editor-data')->setDescription('Check missing editor data. (Filter editorData)');
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('Checking missing editor data...');

		try {
			$dataForSave = $this->editorDataService->runCheck();
			$emails      = Parameters::loadArray('editorData.sendNotificationToIfMissing') ?: [];

			if (!empty($dataForSave) && !empty($emails)) {
				foreach ($this->sites->getSites() as $site) {
					if (!$site->isVisible || !$site->getEmail()) {
						continue;
					}

					$mail = $this->mailBuilderFactory->create($site->getIdent());
					$mail->setSubject('Missing editor data');
					$mail->setTemplateFile(__DIR__ . '/../../AdminModule/Templates/_emails/missingEditorData.latte');
					$mail->setParameters(['data' => $dataForSave]);

					$mail->setFrom($site->getEmail(), $site->getSiteName() ?: '');
					foreach ($emails as $email) {
						$mail->addTo($email);
					}
					$mail->send();

					break;
				}
			}

			$output->writeLn('<info>[OK] - Done</info>');
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 0;
	}
}
