<?php declare(strict_types = 1);

namespace Core\Console\Database;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'database:clear-migrations')]
class ClearMigrations extends Command
{
	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
		parent::__construct();
	}

	protected function configure(): void
	{
		$this->setName('database:clear-migrations')->setDescription('Clear migrations');
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$conn = $this->em->getConnection();

		$conn->executeQuery("ALTER TABLE doctrine_migrations MODIFY `version` VARCHAR(140)");

		$rename = [
			'dynamicmodule'           => 'DynamicModule',
			'eshopcatalog'            => 'EshopCatalog',
			'eshoporders'             => 'EshopOrders',
			'ceskaposta'              => 'Ceskaposta',
			'eshopaccountant'         => 'EshopAccountant',
			'eshopadvancedfeature'    => 'EshopAdvancedFeature',
			'eshopgifts'              => 'EshopGifts',
			'eshopproductscomparison' => 'EshopProductsComparison',
			'eshopsales'              => 'EshopSales',
			'infopanels'              => 'InfoPanels',
			'eshopstatistics'         => 'EshopStatistics',
			'eshopstock'              => 'EshopStock',
			'multihubdropship'        => 'MultihubDropShip',
			'mygls'                   => 'MyGls',
			'eshopcheckout'           => 'EshopCheckout',
			'paypal'                  => 'PayPal',
			'measuringcodes'          => 'MeasuringCodes',
		];

		$output->writeln('CLEAR MIGRATIONS');
		try {
			$migrations = [];
			foreach ($conn->fetchAllAssociative('SELECT version FROM doctrine_migrations') as $row) {
				$migrations[$row['version']] = $row['version'];
			}

			$allFiles = [];
			foreach ([
				         'vendor' => ROOT_DIR . '/vendor/pshkcms/*/src',
				         'custom' => ROOT_DIR . '/custom/*',
			         ] as $dirKey => $baseDir) {
				foreach (glob($baseDir . '/Migrations/*/*/*.php') ?: [] as $file) {
					if ($dirKey === 'custom') {
						$tmp   = explode('custom/', $file);
						$parts = explode('/', $tmp[1]);
					} else {
						$tmp   = explode('pshkcms/', $file);
						$parts = explode('/', $tmp[1]);
					}

					$module   = $rename[$parts[0]] ?? $parts[0];
					$module   = ucfirst($module);
					$basename = basename($file);
					$number   = (int) filter_var($basename, FILTER_SANITIZE_NUMBER_INT);
					$version  = 'Version' . $number;

					$allFiles[$module][] = [
						'basename' => $basename,
						'number'   => $number,
						'version'  => $version,
					];

					$newVersion = $module . "\\Migrations\\Version" . $number;
					if (isset($migrations[$number])) {
						unset($migrations[$number]);
						$conn->executeQuery("UPDATE doctrine_migrations SET version = ? WHERE version = ?", [
							$newVersion, $number,
						]);
					} else if (isset($migrations['Migrations\\' . $version])) {
						unset($migrations['Migrations\\' . $version]);
						$conn->executeQuery("UPDATE doctrine_migrations SET version = ? WHERE version = ?", [
							$newVersion, 'Migrations\\' . $version,
						]);
					} else if (isset($migrations[$newVersion])) {
						unset($migrations[$newVersion]);
					}
				}
			}

			foreach ($migrations as $row) {
				$conn->executeQuery("DELETE FROM doctrine_migrations WHERE version = ?", [$row]);
			}

			$output->writeLn('<info>[OK] - cleared</info>');
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
