<?php declare(strict_types = 1);

namespace Core\Console\Seo;

use Contributte\Translation\Translator;
use Core\FrontModule\Model\SiteMap;
use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Doctrine;
use Exception;
use Nette\Http\Request;
use Nette\Utils\FileSystem;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Thepixeldeveloper\Sitemap\Drivers\XmlWriterDriver;
use Tracy\Debugger;

#[AsCommand(name: 'seo:generate-sitemap')]
class GenerateSiteMap extends Command
{
	public function __construct(
		protected SiteMap    $siteMap,
		protected Request    $httpRequest,
		protected Langs      $langs,
		protected Sites      $sites,
		protected Translator $translator,
	)
	{
		parent::__construct();
	}

	protected function configure(): void
	{
		$this->setName('seo:generate-sitemap')
			->addOption('site', null, InputOption::VALUE_OPTIONAL)
			->addOption('lang', null, InputOption::VALUE_OPTIONAL)
			->setDescription('Generate sitemaps');
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('GENERATE SITEMAP.XML');

		try {
			set_time_limit(3000);
			ini_set('memory_limit', '2G');
			Debugger::dispatch();
			$groups    = ['default'];
			$siteIdent = (string) $input->getOption('site');
			$langValue = (string) $input->getOption('lang');

			$created = [];
			$exist   = [];

			foreach ($this->sites->getSites() as $k => $v) {
				if ($siteIdent && $siteIdent !== $k) {
					continue;
				}

				FileSystem::delete(SiteMap::TMP_DIR . '/' . $k);
				foreach ($v->getDomains() as $domain) {
					if ($domain->disableIndex) {
						@unlink(WWW_DIR . '/sitemap-' . $siteIdent . '-default-' . $langValue . '.xml');

						continue;
					}

					if ($langValue && $langValue !== $domain->getLang()) {
						continue;
					}

					$lang = $domain->getLang();
					/** @phpstan-ignore-next-line */
					Sites::$currentIdentOverride                = $v->getIdent();
					Sites::$currentLangOverride                 = $lang;
					$this->sites->getCurrentSite()->currentLang = $lang;
					$this->translator->setLocale($lang);

					foreach ($groups as $group) {
						$sitemap = $this->siteMap->getSiteMap($lang, $group);

						if (!$sitemap) {
							continue;
						}

						$driver = new XmlWriterDriver();
						$sitemap->accept($driver);

						$filename = "sitemap-$k-$group-$lang.xml";

						$filePath = WWW_DIR . '/' . $filename;

						$created[] = $filePath;
						file_put_contents($filePath, $driver->output());

						if (isset($sitemap->all()[0]) && $sitemap->all()[0] instanceof \Thepixeldeveloper\Sitemap\Sitemap) {
							foreach ($sitemap->all() as $sitemapK => $v) {
								/** @var \Thepixeldeveloper\Sitemap\Sitemap $v */
								$tmp = $this->siteMap->getSiteMap($lang, $group, $sitemapK + 1);

								if (!$tmp) {
									continue;
								}

								$driver = new XmlWriterDriver();
								$tmp->accept($driver);

								$filePath  = WWW_DIR . '/' . basename($v->getLoc());
								$created[] = $filePath;
								file_put_contents($filePath, $driver->output());
								$tmp = null;
							}
						}
						$sitemap = null;
					}
				}
			}

			$output->writeLn('<info>[OK] - sitemaps created</info>');

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
