<?php declare(strict_types = 1);

namespace Core\Console\Utils;

use Core\Model\Parameters;
use Core\Model\SystemConfig;
use Doctrine;
use Exception;
use Nette\Localization\Translator;
use Nette\Mail\Mailer;
use Nette\Mail\Message;
use Nette\Utils\FileSystem;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'core:modify-last-system-update')]
class CoreModifyLastSystemUpdate extends Command
{
	public function __construct(
		protected Mailer     $mailer,
		protected Translator $translator,
	)
	{
		parent::__construct();
	}

	protected function configure(): void
	{
		$this->setName('core:modify-last-system-update')
			->addOption('emailNotify', null, InputOption::VALUE_OPTIONAL, 'Send notify to defined emails', false);
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('<comment>--- Modify last system update ---</comment>');
		$emailNotify = (bool) $input->getOption('emailNotify');

		try {
			FileSystem::write(DATA_DIR . DS . 'core' . DS . 'lastSystemUpdate', (string) time());
			$output->writeLn('<info>[OK] - modified</info>');

			if ($emailNotify) {
				$subject = (string) $this->translator->translate('core.systemUpdate.notifyEmailSubject', ['siteName' => Parameters::load('siteName')]);
				$text    = (string) $this->translator->translate('core.systemUpdate.notifyEmailText');

				foreach ((array) SystemConfig::load('systemUpdateNotifyEmails') ?: [] as $email) {
					try {
						$message = new Message;
						$message->setFrom('noreply@pshk.cz', 'PSHK CMS');
						$message->addTo($email);
						$message->setSubject($subject);
						$message->setHtmlBody($text);

						$this->mailer->send($message);
						$output->writeLn('<info>[OK] - notify sent - ' . $email . '</info>');
					} catch (Exception $e) {
						$output->writeln('<error>[ERROR] - notify failed sent to ' . $email . ' - ' . $e->getMessage() . '</error>');
						Debugger::log($e, '_coreSystemUpdateNotify');
					}
				}
			}
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
			Debugger::log($e, '_coreSystemUpdate');
		}

		return 0;
	}
}
