<?php declare(strict_types = 1);

namespace Core\DI;

use Contributte\DI\Helper\ExtensionDefinitionsHelper;
use Contributte\Scheduler\CallbackJob;
use InvalidArgumentException;
use Nette;
use Nette\DI\Compiler;
use Nette\DI\Definitions\Definition;
use Nette\DI\Definitions\ServiceDefinition;
use Nette\DI\Definitions\Statement;
use Nette\DI\InvalidConfigurationException;
use Nette\Neon\Neon;
use Nette\PhpGenerator\ClassType;
use Nette\Utils\FileSystem;
use Nette\Utils\Finder;
use Nette\Utils\Strings;
use Nettrine\ORM\DI\Helpers\MappingHelper;
use ReflectionClass;
use Tracy\Debugger;

abstract class CompilerExtension extends Nette\DI\CompilerExtension
{
	protected static int $schedulerJobKey = 0;

	protected bool    $doctrineAttributes = true;
	protected ?string $extensionName      = null;
	protected ?string $cDIDir             = null;
	protected array   $decoratorConfig    = [];

	public function setCompiler(Compiler $compiler, string $name): static
	{
		parent::setCompiler($compiler, $name);

		FileSystem::createDir(TMP_DIR);
		FileSystem::createDir(LOG_DIR);
		FileSystem::createDir(UPLOADS_DIR);
		FileSystem::createDir(THUMBS_DIR);

		$extensionsFile = $this->getDIDir() . '/extensions.neon';
		if (file_exists($extensionsFile)) {
			$fileContent = file_get_contents($extensionsFile);

			if ($fileContent) {
				$extensions = Neon::decode($fileContent);

				foreach ($extensions ?? [] as $extName => $class) {
					$args = [];
					if ($class instanceof Statement) {
						[$class, $args] = [$class->getEntity(), $class->arguments];
					}

					if (!is_a($class, Nette\DI\CompilerExtension::class, true)) {
						throw new InvalidConfigurationException(
							"Extension '$class' not found or is not Nette\\DI\\CompilerExtension descendant."
						);
					}

					$this->compiler->addExtension($extName, (new ReflectionClass($class))->newInstanceArgs($args));
				}
			}
		}

		return $this;
	}

	public function loadConfiguration(): void
	{
		parent::loadConfiguration();

		$builder = $this->getContainerBuilder();

		if (file_exists($this->getDIDir() . '/../Migrations')) {
			$migrationsConfiguration = $builder->getDefinition('migrations.configuration');
			$migrationsConfiguration
				->addSetup(
					'addMigrationsDirectory',
					[$this->getExtensionName() . '\\Migrations', $this->getDIDir() . '/../Migrations'],
				);
		}
	}

	public function beforeCompile(): void
	{
		parent::beforeCompile();
		$builder = $this->getContainerBuilder();
		$config  = $this->getConfig();

		$translateResources = $this->getTranslationResources();
		if ($translateResources) {
			/** @var ServiceDefinition $translator */
			$translator = $builder->getDefinition('translation.translator');

			/** @var ServiceDefinition $neonDoctrineLoader */
			$neonDoctrineLoader = $builder->getDefinition('core.staticTexts');

			if (!Debugger::$productionMode) {
				$tracyPanel = $builder->getDefinition('translation.tracyPanel');
			}

			foreach ($translateResources as $k1 => $v1) {
				if (!file_exists($v1)) {
					continue;
				}

				foreach (Finder::find('*.neon')->from($v1) as $v2) {
					$match = Strings::match(
						$v2->getFilename(),
						'~^(?P<domain>.*?)\.(?P<locale>[^\.]+)\.(?P<format>[^\.]+)$~',
					);

					if (!$match) {
						continue;
					}

					$translator->addSetup('addResource', [$match['format'], $v2->getPathname(), $match['locale'],
						$match['domain']]);

					$neonDoctrineLoader->addSetup('addResource', [$match['locale'], $match['domain'],
						$v2->getPathname()]);

					if (isset($tracyPanel)) {
						$tracyPanel->addSetup('addResource', [$match['format'], $v2->getPathname(), $match['locale'],
							$match['domain']]);
					}
				}
			}
		}

		$extName = $this->getExtensionName();
		if ($extName !== 'Core') {
			$ref = new ReflectionClass(static::class);
			$dir = dirname((string) $ref->getFileName()) . '/../Model/Entities';

			if (file_exists($dir)) {
				MappingHelper::of($this)
					->addAttribute('default', $extName . '\Model\Entities', $dir);
			}
		}

		if (isset(((array) $config)['scheduler']['jobs'])) {
			$definitionHelper = new ExtensionDefinitionsHelper($this->compiler);
			/** @var ServiceDefinition $schedulerDefinition */
			$schedulerDefinition = $builder->getDefinition('scheduler.scheduler');

			foreach (((array) $config)['scheduler']['jobs'] as $jobName => $jobConfig) {
				if (is_numeric($jobName)) {
					$jobName = self::$schedulerJobKey;
					self::$schedulerJobKey++;
				}

				if (is_array($jobConfig) && (isset($jobConfig['cron']) || isset($jobConfig['callback']))) {
					if (!isset($jobConfig['cron'], $jobConfig['callback'])) {
						throw new InvalidArgumentException(
							sprintf(
								'Both options "callback" and "cron" of %s > jobs > %s must be configured',
								$this->name,
								$jobName,
							)
						);
					}

					$jobDefinition = new Statement(CallbackJob::class, [$jobConfig['cron'], $jobConfig['callback']]);
				} else {
					$jobPrefix     = $this->prefix('job.' . $jobName);
					$jobDefinition = $definitionHelper->getDefinitionFromConfig($jobConfig, $jobPrefix);
					if ($jobDefinition instanceof Definition) {
						$jobDefinition->setAutowired(false);
					}
				}

				$schedulerDefinition->addSetup('add', [$jobDefinition, $jobName]);
			}
		}
	}

	public function afterCompile(ClassType $class): void
	{
		parent::afterCompile($class);
		$init = $class->getMethod('initialize');

		$ref = new ReflectionClass(static::class);

		$dir = dirname((string) $ref->getFileName()) . '/../';
		$init->addBody(
			sprintf('define("PACKAGE_%s_DIR", \'%s\');', strtoupper((string) $this->getExtensionName()), $dir),
		);
	}

	protected function setMapping(array $mapping): void
	{
		$builder = $this->getContainerBuilder();

		/** @var ServiceDefinition $presenterFactory */
		$presenterFactory = $builder->getDefinition('nette.presenterFactory');
		$presenterFactory->addSetup('setMapping', [$mapping]);
	}

	public function getTranslationResources(): array
	{
		return [$this->getDIDir() . DS . '..' . DS . 'lang'];
	}

	private function getExtensionName(): string
	{
		if (!$this->extensionName) {
			$this->extensionName = explode('\\', static::class)[0];
		}

		return $this->extensionName;
	}

	protected function getDIDir(): ?string
	{
		if ($this->cDIDir === null) {
			$obj          = new ReflectionClass($this);
			$this->cDIDir = dirname((string) $obj->getFileName());
		}

		return $this->cDIDir;
	}
}
