<?php declare(strict_types = 1);

namespace Core\Model\Application;

use Core\Model\Countries;
use Core\Model\Helpers\CoreHelper;
use Core\Model\Http\Session;
use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Currency\Model\Config as CurrencyConfig;
use Currency\Model\Currencies;
use Navigations\Model\NavigationConfig;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;
use Nette\Utils\Json;

class AppState
{
	final public const CACHE_NAMESPACE = 'appState';
	protected Cache $cache;

	protected static array   $state           = [];
	protected static ?string $uniqueRequestId = null;
	protected static array   $jsAssets        = [];

	public function __construct(
		protected Sites     $sites,
		protected Session   $session,
		protected Request   $request,
		protected Response  $response,
		Storage             $storage,
		protected Countries $countries,
		protected Container $container,
	)
	{
		$this->cache = new Cache($storage, self::CACHE_NAMESPACE);
	}

	public function init(): void
	{
		$currentSite = $this->sites->getCurrentSite();

		self::setState('currentSite', $currentSite);

		if (php_sapi_name() === 'cli') {
			return;
		}

		$transfer = $this->request->getQuery('transfer');

		if ($transfer) {
			$json = base64_decode(strtr($transfer, '-_', '+/'));
			$data = Json::decode($json, forceArrays: true);

			if (
				isset($data['sid'], $data['ua'], $data['ts'], $data['sig'], $data['url'])
				&& abs(time() - $data['ts']) < 30
				&& $data['ip'] === CoreHelper::ipPrefix($_SERVER['REMOTE_ADDR'] ?? '')
			) {
				$sig = $data['sig'];
				unset($data['sig']);

				$userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
				$secret    = (string) Parameters::loadScalar('system.transferToken');

				if (strpos(sha1($userAgent), $data['ua']) === 0) {
					$expectedSig = hash_hmac('sha256', Json::encode($data), $secret);

					if (hash_equals($expectedSig, $sig)) {
						$this->response->setCookie('PHPSESSID', $data['sid'], (new \DateTime())->modify('+1 month')->getTimestamp());
					}
				}
			}

			$url = $this->request->getUrl();
			$url = $url->withQueryParameter('transfer', null);

			$this->response->redirect($url->getAbsoluteUrl());
		}

		$cookieBarEnabled = SystemConfig::load('cookieBar.enable');
		$isCookieBarOpen  = false;
		if ($cookieBarEnabled) {
			$cookieBarKey    = SystemConfig::loadString('cookieBar.key');
			$cookieBarValues = $this->request->getCookie($cookieBarKey);

			$allowCookieServices = [];
			$allowCookieSections = [];
			$isCookieBarOpen     = $cookieBarValues === null;
			/** @var array{type: string, services: array<string>, files: array<string>} $cookie */
			foreach ($cookieBarValues ? Json::decode($cookieBarValues, forceArrays: true) : [] as $cookie) {
				$allowCookieSections[] = $cookie['type'];
				foreach ($cookie['services'] as $service) {
					$allowCookieServices[] = $service;
				}
			}

			self::setState('allowCookieServices', $allowCookieServices);
			self::setState('allowCookieSections', $allowCookieSections);
		}
		self::setState('isCookieBarOpen', $isCookieBarOpen);

		$queryLang = $this->request->getQuery('lang');
		if ($queryLang) {
			$currentSite->currentLang = $queryLang;
		}

		$domain = $currentSite->getCurrentDomain();

		if (!$domain) {
			return;
		}

		if (!$this->request->getCookie($this->getCountryCookieName()) && $domain->defaultCountry) {
			$this->setCountry($domain->defaultCountry);
		}

		if (
			$this->container->hasService('currency.currencies')
			&& class_exists('Currency\Model\Currencies')
			&& class_exists('Currency\Model\Config')
		) {
			/** @var Currencies $currencies */
			$currencies = $this->container->getService('currency.currencies');
			$currency   = $currencies->getCurrent();

			if (!empty(CurrencyConfig::load('siteAllowedCurrencies'))) {
				$allowedCurrencies = CurrencyConfig::loadArray('siteAllowedCurrencies.' . $domain->site->getIdent() . '.' . $domain->getLang());

				if (!empty($allowedCurrencies) && !Arrays::contains($allowedCurrencies, $currency->getCode())) {
					//					foreach (CurrencyConfig::load('siteAllowedCurrencies.' . $domain->site->getIdent()) ?? [] as $siteLang => $siteCurrencies) {
					//						if (Arrays::contains($siteCurrencies, $currency->getCode())) {
					//							exit;
					//						}
					//					}

					$currency = $currencies->getActive()[$allowedCurrencies[0]];
					$currencies->setCurrentCurrency($currency);
				}
			}

			self::setState('currency', $currency);

			if (CurrencyConfig::load('changeCountry') && count($currencies->getAll()) > 1) {
				$country = CurrencyConfig::load('changeCountry')[$currencies->getCurrent()->getCode()] ?? null;

				if ($country && $c = $this->countries->checkId($country)) {
					$this->setCountry($c);
				}
			}
		}
	}

	protected function getCountryCookieName(): string
	{
		$name   = 'country';
		$suffix = null;
		$domain = $this->sites->getCurrentSite()->getCurrentDomain();
		if (NavigationConfig::load('showDefaultLangInUrl') || !$domain->isDefault) {
			$suffix = $domain->getLang();
		}

		return $name . ($suffix ? ucfirst($suffix) : '');
	}

	public function setCountry(string $country): void
	{
		$key = $this->getCountryCookieName();

		if (!empty(SystemConfig::load('siteAllowedCountries', []))) {
			$domain = $this->sites->getCurrentSite()->getCurrentDomain();

			if ($domain) {
				$allowedCountries = (array) SystemConfig::load(
					'siteAllowedCountries.' . $domain->site->getIdent() . '.' . $domain->getLang(),
				) ?: [];

				if (!empty($allowedCountries) && !Arrays::contains($allowedCountries, $country)) {
					$country = $allowedCountries[0];
				}
			}
		}

		self::setState('freshCountrySet', $country);
		$this->response->setCookie($key, $country, (new DateTime())->modify('+1 year'));
	}

	public function getCountry(): ?string
	{
		if (self::getState('freshCountrySet')) {
			return self::getState('freshCountrySet');
		}

		return $this->request->getCookie($this->getCountryCookieName()) ?: $this->sites->getCurrentSite()
			->getCurrentDomain()->defaultCountry;
	}

	public static function setState(string $key, mixed $value): void
	{
		self::$state[$key] = $value;
	}

	/**
	 * @return mixed|null
	 */
	public static function getState(string $key, mixed $default = null)
	{
		return self::$state[$key] ?? $default;
	}

	public static function getUniqueRequestId(string $salt = ''): string
	{
		if (self::$uniqueRequestId === null) {
			self::$uniqueRequestId = md5(uniqid($salt, true) . time());
		}

		return self::$uniqueRequestId;
	}

	public static function addJs(string $key, string $url): void { self::$jsAssets[$key] = $url; }

	public static function getJs(): array { return self::$jsAssets; }

	/**
	 * @param string|int $id
	 */
	public static function addUpdatedEntityId(string $key, $id): void
	{
		if (!isset(self::$state['updatedEntities'][$key])) {
			self::$state['updatedEntities'][$key] = [];
		}

		self::$state['updatedEntities'][$key][] = $id;
	}

	public static function getUpdatedEntitiesIds(string $key): array
	{
		return self::$state['updatedEntities'][$key] ?? [];
	}
}
