<?php declare(strict_types = 1);

namespace Core\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Contributte\Translation\Translator;
use Core\Model\Dao;
use Core\Model\Entities\Country;
use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query\Expr\Join;
use Nette\Caching\Cache;

/**
 * @method Country|null getReference($id)
 */
class Countries extends BaseEntityService
{
	final public const CACHE_NAMESPACE = 'countries';

	/** @var string */
	protected $entityClass = Country::class;

	protected ?array $cOptionsForSelect = null;

	/** @var Dao\Country[][] */
	protected array  $cDao = [];
	protected ?array $cAll = null;

	public function __construct(
		protected Translator $translator
	)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @param string $id
	 */
	public function get($id): ?Country
	{
		$countries = [];
		foreach ($this->getAll() as $country) {
			$countries[strtolower((string) $country->getId())] = $country;
			$countries[strtoupper((string) $country->getId())] = $country;
		}

		return $countries[$id] ?? null;
	}

	public function checkId(string $id): ?string
	{
		foreach ($this->getDao() as $countryId => $country) {
			if (strtolower((string) $id) === $countryId || strtoupper((string) $id) === $countryId) {
				return $countryId;
			}
		}

		return null;
	}

	public function setPosition(string $id, int $position): bool
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/** @return Country[]|null */
	public function getAll(): ?array
	{
		if ($this->cAll === null) {
			$countriesQuery = $this->getEr()->createQueryBuilder('c', 'c.id');
			$countriesQuery->orderBy('c.position');

			$this->cAll = $countriesQuery->getQuery()->getResult();
		}

		return $this->cAll;
	}

	/**
	 * @return Dao\Country[]
	 */
	public function getDao(): array
	{
		$lang = $this->translator->getLocale();
		if (!array_key_exists($lang, $this->cDao)) {
			$this->cDao[$lang] = $this->getCache()->load(self::CACHE_NAMESPACE . '/' . $lang, function(&$dep) use ($lang) {
				$dep = [
					Cache::Expire => '1 month',
				];

				$data = [];
				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select('c.id, c.iso3166_1, t.name')
					         ->innerJoin('c.texts', 't', Join::WITH, 't.lang = :lang')
					         ->setParameters(new ArrayCollection([new Parameter('lang', $lang)]))
					         ->orderBy('c.position')->getQuery()
					         ->getArrayResult() as $row) {
					$data[$row['id']] = new Dao\Country(
						(string) $row['id'],
						(string) $row['iso3166_1'],
						(string) $row['name'],
					);
				}

				return $data;
			});
		}

		return $this->cDao[$lang];
	}

	public function getAllNameColumn(): array
	{
		if ($this->cOptionsForSelect === null) {
			$this->cOptionsForSelect = [];

			foreach ($this->getDao() as $row) {
				$this->cOptionsForSelect[$row->getId()] = $row->getName();
			}
		}

		return $this->cOptionsForSelect;
	}
}
