<?php declare(strict_types = 1);

namespace Core\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table(name: 'core__country')]
#[ORM\Entity]
#[ORM\EntityListeners([TranslateListener::class])]
class Country
{
	use TTranslateListener;

	#[ORM\Id]
	#[ORM\Column(name: '`id`', type: Types::STRING, length: 255, nullable: false)]
	public string $id;

	/**
	 * @var Collection<string, CountryText>
	 */
	#[ORM\OneToMany(mappedBy: 'country', targetEntity: CountryText::class, indexBy: 'lang')]
	public Collection $texts;

	#[ORM\Column(name: 'iso3166_1', type: Types::STRING, length: 3, nullable: false, options: ['default' => 'CZE'])]
	protected string $iso3166_1 = 'CZE';

	#[Gedmo\SortablePosition]
	#[ORM\Column(name: 'position', type: 'integer')]
	protected int $position = 0;

	public function __construct(string $id, string $iso3166_1)
	{
		$this->id        = $id;
		$this->iso3166_1 = $iso3166_1;
		$this->setPosition(-1);
		$this->texts = new ArrayCollection();
	}

	public function getId(): string { return $this->id; }

	public function getText(?string $lang = null): ?CountryText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/** @return Collection<string, CountryText> */
	public function getTexts() { return $this->texts; }

	public function getPosition(): int { return $this->position; }

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function getIso3166_1(): string { return strtoupper($this->iso3166_1); }

	public function setIso3166_1(string $iso3166_1): self
	{
		$this->iso3166_1 = $iso3166_1;

		return $this;
	}
}
