<?php declare(strict_types = 1);

namespace Core\Model\Entities;

use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\ORM\Internal\Hydration\IterableResult;

class QueryBuilder extends \Doctrine\ORM\QueryBuilder
{
	private array $criteriaJoins = [];

	public function whereCriteria(array $criteria): self
	{
		foreach ($criteria as $key => $val) {
			$alias = $this->autoJoin($key);

			$operator = '=';
			if (preg_match('~(?P<key>[^\\s]+)\\s+(?P<operator>.+)\\s*~', (string) $key, $m)) {
				$key      = $m['key'];
				$operator = strtr(strtolower($m['operator']), [
					'neq' => '!=',
					'eq'  => '=',
					'lt'  => '<',
					'lte' => '<=',
					'gt'  => '>',
					'gte' => '>=',
				]);
			}

			$not = str_starts_with($operator, '!');
			if (str_starts_with($operator, 'not')) {
				$operator = substr($operator, 4);
				$not      = true;
			}

			$paramName = 'param_' . (count($this->getParameters()) + 1);

			if (is_array($val)) {
				$this->andWhere("$alias.$key " . ($not ? 'NOT ' : '') . "IN (:$paramName)");
				$this->setParameter(
					$paramName,
					$val,
					is_integer(reset($val)) ? ArrayParameterType::INTEGER : ArrayParameterType::STRING,
				);
			} else if ($val === null) {
				$this->andWhere("$alias.$key IS " . ($not ? 'NOT ' : '') . 'NULL');
			} else {
				$this->andWhere(sprintf('%s.%s %s :%s', $alias, $key, strtoupper($operator), $paramName));
				$this->setParameter($paramName, $val);
			}
		}

		return $this;
	}

	public function autoJoinOrderBy(array|string $sort, ?string $order = null): self
	{
		if (is_array($sort)) {
			foreach (func_get_arg(0) as $k => $v) {
				if (!is_string($k)) {
					$k = $v;
					$v = null;
				}
				$this->autoJoinOrderBy($k, $v);
			}

			return $this;
		}

		$reg = '~[^()]+(?=\))~';
		if (preg_match($reg, $sort, $matches)) {
			$sortMix     = $sort;
			$sort        = $matches[0];
			$alias       = $this->autoJoin($sort, 'leftJoin');
			$hiddenAlias = $alias . $sort . (is_countable($this->getDQLPart('orderBy')) ? count(
					$this->getDQLPart('orderBy'),
				) : 0);

			$this->addSelect(preg_replace($reg, $alias . '.' . $sort, $sortMix) . ' as HIDDEN ' . $hiddenAlias);
			$rootAliases = $this->getRootAliases();
			$this->addGroupBy(reset($rootAliases) . '.id');
			$sort = $hiddenAlias;
		} else {
			$alias = $this->autoJoin($sort);
			$sort  = $alias . '.' . $sort;
		}

		return $this->addOrderBy($sort, $order);
	}

	private function autoJoin(string &$key, string $methodJoin = "innerJoin"): string
	{
		$rootAliases = $this->getRootAliases();
		$alias       = reset($rootAliases);

		if (($i = strpos($key, '.')) === false || !in_array(substr($key, 0, $i), $rootAliases)) {
			// there is no root alias to join from, assume first root alias
			$key = $alias . '.' . $key;
		}

		while (preg_match('~([^\\.]+)\\.(.+)~', $key, $m)) {
			$key      = $m[2];
			$property = $m[1];

			if (in_array($property, $rootAliases)) {
				$alias = $property;
				continue;
			}

			if (isset($this->criteriaJoins[$alias][$property])) {
				$alias = $this->criteriaJoins[$alias][$property];
				continue;
			}

			$j = 0;
			do {
				$joinAs = substr($property, 0, 1) . $j++;
			} while (isset($this->criteriaJoins[$joinAs]));
			$this->criteriaJoins[$joinAs] = [];

			$this->{$methodJoin}("$alias.$property", $joinAs);
			$this->criteriaJoins[$alias][$property] = $joinAs;
			$alias                                  = $joinAs;
		}

		return $alias;
	}
}
