<?php declare(strict_types = 1);

namespace Core\Model\Helpers;

use DateTimeImmutable;
use DateTimeInterface;
use InvalidArgumentException;

final class Caster
{
	public static function toInt(mixed $value): int
	{
		if (is_int($value)) {
			return $value;
		}
		if (is_numeric($value) && preg_match('/^-?\d+$/', (string) $value) === 1) {
			return (int) $value;
		}
		throw new InvalidArgumentException("Nelze převést hodnotu na int: " . get_debug_type($value));
	}

	public static function toFloat(mixed $value): float
	{
		if (is_float($value)) {
			return $value;
		}
		if (is_int($value) || is_numeric($value)) {
			return (float) $value;
		}
		throw new InvalidArgumentException("Nelze převést hodnotu na float: " . get_debug_type($value));
	}

	public static function toBool(mixed $value): bool
	{
		if (is_bool($value)) {
			return $value;
		}

		if (is_string($value)) {
			$normalized = strtolower(trim($value));

			return match ($normalized) {
				'1', 'true', 'yes', 'on' => true,
				'0', 'false', 'no', 'off', '' => false,
				default => throw new InvalidArgumentException("Nelze převést hodnotu na bool: " . $value),
			};
		}

		if (is_int($value)) {
			return $value !== 0;
		}

		throw new InvalidArgumentException("Nelze převést hodnotu na bool: " . get_debug_type($value));
	}

	public static function toString(mixed $value): string
	{
		if (is_string($value)) {
			return $value;
		}
		if (is_scalar($value) || (is_object($value) && method_exists($value, '__toString'))) {
			return (string) $value;
		}
		throw new InvalidArgumentException("Nelze převést hodnotu na string: " . get_debug_type($value));
	}

	/**
	 * @return list<mixed>|array<string, mixed>
	 */
	public static function toArray(mixed $value): array
	{
		if (is_array($value)) {
			return $value;
		}
		if ($value instanceof \Traversable) {
			return iterator_to_array($value);
		}
		throw new InvalidArgumentException("Nelze převést hodnotu na array: " . get_debug_type($value));
	}

	public static function toDateTimeImmutable(mixed $value): DateTimeImmutable
	{
		if ($value instanceof DateTimeImmutable) {
			return $value;
		}

		if ($value instanceof DateTimeInterface) {
			return DateTimeImmutable::createFromInterface($value);
		}

		if (is_string($value)) {
			$dt = DateTimeImmutable::createFromFormat(DateTimeInterface::ATOM, $value)
				?: new DateTimeImmutable($value);

			return $dt;
		}

		throw new InvalidArgumentException("Nelze převést hodnotu na DateTimeImmutable: " . get_debug_type($value));
	}
}
