<?php declare(strict_types = 1);

namespace Core\Model\Images\Macros;

use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\SystemConfig;
use Latte\CompileException;
use Latte\Compiler;
use Latte\MacroNode;
use Latte\Macros\MacroSet;
use Latte\PhpWriter;
use Latte\Runtime\Template;
use Nette;
use Nette\InvalidStateException;

class Macros extends MacroSet
{
	private bool $isUsed = false;

	public function __construct(
		Compiler $compiler,
	)
	{
		parent::__construct($compiler);
	}

	public static function install(Compiler $compiler): self
	{
		$me = new self($compiler);

		/**
		 * {img [namespace/]$name[, $size[, $flags]]}
		 */
		$me->addMacro('img', $me->macroImg(...), null, $me->macroAttrImg(...));
		$me->addMacro('imgLazy', $me->macroImgLazy(...), null, $me->macroAttrImgLazy(...));

		return $me;
	}

	public function macroImg(MacroNode $node, PhpWriter $writer): string
	{
		$data = $this->prepareMacroImg($node, $writer);

		$return = $writer->write('echo %escape(' . $writer->formatWord($data['src']) . ')');

		if (isset($data['width'], $data['height']) && $node->htmlNode && $node->htmlNode->name === 'img') {
			$return .= '.(\Core\Model\SystemConfig::load(\'images.autoGeneratedSize\', false) ? "\" width=\"' . $writer->formatWord((string) $data['width']) . '\" height=\"' . $writer->formatWord((string) $data['height']) . '" : \'\')';
		}

		return $return;
	}

	public function macroAttrImg(MacroNode $node, PhpWriter $writer): string
	{
		$data = $this->prepareMacroAttrImg($node, $writer);

		$return = '?>';
		$return .= $writer->write(' src="<?php echo %escape(' . $writer->formatWord($data['src']) . ')?>"');

		if ($data['srcset']) {
			$return .= $writer->write(' srcset="<?php ' . $data['srcset'] . ' ?>"');
		}

		$return .= $writer->write('
			<?php if (\Core\Model\SystemConfig::load("images.autoGeneratedSize", false)): ?>
				width="<?php echo %escape(' . $writer->formatWord((string) $data['width']) . ')?>"
				height="<?php echo %escape(' . $writer->formatWord((string) $data['height']) . ')?>"
			<?php endif ?>
		');

		$return .= '<?php';

		return $return;
	}

	public function macroImgLazy(MacroNode $node, PhpWriter $writer): string
	{
		$data = $this->prepareMacroImg($node, $writer);

		$return = $writer->write('
			<?php if (\Core\Model\SystemConfig::load("images.autoGeneratedSize", false)): ?>
				width="<?php echo ' . $writer->formatWord((string) $data['width']) . ' ?>"
				height="<?php echo ' . $writer->formatWord((string) $data['height']) . ' ?>"
			<?php endif ?>
		');

		if (SystemConfig::load('images.lazyLoading', false)) {
			$return .= ' loading="lazy"';
		}

		$return .= $writer->write(' src="<?php echo (' . $writer->formatWord($data['src']) . ') ?>"<?php');

		return $return;
	}

	public function macroAttrImgLazy(MacroNode $node, PhpWriter $writer): string
	{
		if (SystemConfig::load('images.lazyLoading') !== true) {
			return $this->macroAttrImg($node, $writer);
		}

		$data = $this->prepareMacroAttrImg($node, $writer);

		$return = '?>';
		$return .= $writer->write(' src="<?php echo %escape(' . $writer->formatWord($data['src']) . ')?>"');

		if ($data['srcset']) {
			$return .= $writer->write(' srcset="<?php ' . $data['srcset'] . ' ?>"');
		}

		if (SystemConfig::loadScalar('images.lazyLoading')) {
			$return .= ' loading="lazy"';
		}

		$return .= $writer->write('<?php if (\Core\Model\SystemConfig::load("images.autoGeneratedSize", false)): ?>');

		if (isset($data['width'], $data['height'])) {
			$return .= $writer->write('
				width="<?php echo %escape(' . $writer->formatWord((string) $data['width']) . ')?>"
				height="<?php echo %escape(' . $writer->formatWord((string) $data['height']) . ')?>"');
		} else {
			$return .= $writer->write('
				width="<?php echo explode("x", ' . $data['size'] . ')[0] ?>"
				height="<?php echo explode("x", ' . $data['size'] . ')[1] ?>"');
		}

		$return .= '<?php endif ?><?php';

		return $return;
	}

	protected function prepareMacroImg(MacroNode $node, PhpWriter $writer): array
	{
		$this->isUsed = true;
		$arguments    = Helpers::prepareMacroArguments($node->args);
		if ($arguments["name"] === null) {
			throw new CompileException("Please provide filename.");
		}

		$namespace = $arguments["namespace"];
		unset($arguments["namespace"]);
		$arguments = array_map(function($value) use ($writer) {
			return $value ? $writer->formatWord($value) : 'NULL';
		}, $arguments);

		$command = '($_imagePipe ?: $this->global->imagePipe)';
		$command .= $namespace !== null ? '->setNamespace(' . $writer->formatWord(trim((string) $namespace)) . ')' : '';
		$command .= '->request(' . implode(", ", $arguments) . ')';

		$return = [
			'src' => $command,
		];

		if (isset($arguments['size']) && Nette\Utils\Strings::contains($arguments['size'], 'x')) {
			[$w, $h] = explode('x', str_replace(["'", '"'], '', $arguments['size']));
			$return['placeholderWidth']  = $w;
			$return['placeholderHeight'] = $h;
			$return['width']             = $w;
			$return['height']            = $h;
		}

		return $return;
	}

	protected function prepareMacroAttrImg(MacroNode $node, PhpWriter $writer): array
	{
		$this->isUsed = true;
		$arguments    = Helpers::prepareMacroArguments($node->args);
		if ($arguments["name"] === null) {
			throw new CompileException("Please provide filename.");
		}

		unset($arguments["namespace"]);
		[$width, $height] = isset($arguments['size']) && !Strings::contains($arguments['size'], '$')
			? array_map('intval', explode('x', $arguments['size'])) : [null, null];
		$arguments = array_map(function($value) use ($writer) {
			return $value ? $writer->formatWord($value) : 'NULL';
		}, $arguments);

		$srcset = '[$maxWidth, $maxHeight] = array_map(\'intval\', explode(\'x\', ' . $arguments['size'] . '));' . PHP_EOL
			. '$skip = false;' . PHP_EOL
			. '$lastKey = array_key_last(\Core\Model\SystemConfig::load(\'images.srcsetSizes\'));' . PHP_EOL
			. 'foreach(\Core\Model\SystemConfig::load(\'images.srcsetSizes\') as $k => $srcSize) {' . PHP_EOL
			. sprintf('$imgSize = \Core\Model\Images\Macros\Helpers::calculateNewSizeFromString((string) %s, $srcSize[\'w\']);', $arguments['size']) . PHP_EOL
			. sprintf('if ($srcSize[\'w\'] > $maxWidth) {$skip = true;}') . PHP_EOL
			. sprintf('if (%s === \'fit\') {$imgSize[\'height\'] = 0;}', $arguments['flags']) . PHP_EOL
			. sprintf('echo $_imagePipe->request(%s, $imgSize[\'width\'] . \'x\' . $imgSize[\'height\'], %s, %s, %s, %s, %s) . " " . $srcSize[\'w\'] . "w";',
				$arguments['name'], $arguments['flags'], $arguments['watermark'], $arguments['strictMode'], $arguments['outputExtension'], $arguments['quality']
			) . PHP_EOL
			. 'if ($skip) {break;}' . PHP_EOL
			. 'else { echo ($lastKey !== $k ? "," : ""); }' . PHP_EOL
			. '}' . PHP_EOL;

		$command = '$_imagePipe->request(' . implode(", ", $arguments) . ')';

		$return = [
			'srcset'            => $srcset,
			'src'               => $command,
			'placeholderWidth'  => $width,
			'placeholderHeight' => $height,
			'size'              => $arguments['size'],
			'width'             => isset($arguments['size']) && Strings::contains($arguments['size'], '$')
				? "explode('x', {$arguments['size']})[0]" : $width,
			'height'            => isset($arguments['size']) && Strings::contains($arguments['size'], '$')
				? "explode('x', {$arguments['size']})[1]" : $height,
		];

		return $return;
	}

	public function initialize(): void { $this->isUsed = false; }

	public function finalize(): ?array
	{
		if (!$this->isUsed) {
			return null;
		}

		return [
			get_called_class() . '::validateTemplateParams($template);',
		];
	}

	/**
	 * @param Template|Nette\Application\UI\Template $template
	 */
	public static function validateTemplateParams($template): void
	{
		if (!$template || (is_object($template) && !method_exists($template, 'getParameters'))) {
			return;
		}

		$params = $template->getParameters();
		if (!isset($params['_imagePipe']) || !$params['_imagePipe'] instanceof ImagePipe) {
			$where = isset($params['control']) ?
				" of component " . get_class($params['control']) . '(' . $params['control']->getName() . ')'
				: null;

			throw new InvalidStateException(
				'Please provide an instanceof Img\\ImagePipe ' .
				'as a parameter $_imagePipe to template' . $where
			);
		}
	}

	public function setParams(array $params): void { }
}
