<?php declare(strict_types = 1);

namespace Core\Model\Templating\Filters;

use Contributte\Translation\Translator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\CssHelper;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\Images\Macros\Helpers;
use Core\Model\SystemConfig;
use Nette\Utils\Image;
use voku\helper\HtmlDomParser;

class EditorData
{
	public static ?string $maxWidth = null;

	public function __construct(
		protected Translator      $translator,
		protected ImagePipe       $imagePipe,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	/**
	 * @return string|null
	 */
	public function __invoke(?string $string): ?string
	{
		if (!$string) {
			return $string;
		}

		$html = HtmlDomParser::str_get_html($string);

		$eventData = [
			'html' => $html,
		];
		$this->eventDispatcher->dispatch(new Event($eventData), 'editorData.processHtml');

		$readMore = $html->find('.ck-readmore');

		$mText = $this->translator->translate('front.editor.readMore');
		$lText = $this->translator->translate('front.editor.readLess');
		foreach ($readMore as $k => $rm) {
			$prev   = $rm->prev_sibling();
			$method = $prev->tag != 'p' ? 'outertext' : 'innertext';

			$prev->$method .= ' <a href="#readMore' . $k . '" class="ck-readmore-trigger" data-more="' . $mText . '" data-less="' . $lText . '">' . $mText . '</a>';
			$rm->id        = 'readMore' . $k;
		}

		$string = (string) $html;
		$string = str_replace(["\r\n", "\r", "\n"], '', $string);

		$eventData = [
			'string' => &$string,
		];
		$this->eventDispatcher->dispatch(new Event($eventData), 'editorData.processString');

		// original image replace by image preview
		$lazyLoading = SystemConfig::load('images.lazyLoading', false);
		$srcsetSizes = (array) SystemConfig::load('images.srcsetSizes') ?: [];
		foreach ($html->find('img') as $el) {
			$src   = trim(str_replace('%20', ' ', $el->getAttribute('src')));
			$sizes = CssHelper::parse($el->getAttribute('style'));

			foreach (['width', 'height'] as $s) {
				if (isset($sizes[$s])) {
					$sizes[$s] = (int) str_replace('px', '', (string) $sizes[$s]);
				} else {
					$sizes[$s] = 0;
				}
			}

			$width  = $sizes['width'];
			$height = $sizes['height'];

			if (self::$maxWidth && $width > self::$maxWidth) {
				$width = self::$maxWidth;
			}

			if ((!$width || !$height) && Strings::startsWith($src, '/')) {
				$tmp = @getimagesize(WWW_DIR . $src);
				if ($tmp) {
					[$width, $height] = $tmp;
				}
			}

			$srcset = [];
			foreach ($srcsetSizes as $k => $sizes) {
				$newWidth = $sizes['w'];

				if (!$width || !$height || (isset($srcsetSizes[$k - 1]) && $srcsetSizes[$k - 1]['w'] > $width)) {
					break;
				}

				$newSize  = Helpers::calculateNewSize((int) $width, (int) $height, (int) $newWidth);
				$srcset[] = $this->imagePipe->request(
						$src,
						"{$newSize['width']}x{$newSize['height']}",
						Image::FIT,
					) . ' ' . $newWidth . 'w';
			}

			$src = $this->imagePipe->request($src, "{$width}x{$height}", Image::FIT);
			if (!$width) {
				$imageSize = @getimagesize($src);

				if (isset($imageSize[0]) && $imageSize[0] > 0) {
					$el->setAttribute('style', $el->getAttribute('style') . ';width:' . $imageSize[0] . 'px');
				}
			}

			if ($lazyLoading) {
				$el->setAttribute('loading', 'lazy');
			}

			$el->setAttribute('src', $src);
			if ($srcset) {
				$el->setAttribute('srcset', implode(',', $srcset));
			}
		}

		foreach ($html->find('table') as $table) {
			$table->outerhtml = '<div class="table-wrap">' . $table->outertext . '</div>';
		}

		return (string) $html->outerHtml;
	}
}
