<?php declare(strict_types = 1);

namespace Core\Model\Translation;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Parameters;
use DeepL\Translator;
use DeepL\Usage;
use Exception;
use Nette\Utils\Validators;
use Tracy\Debugger;
use Users\Model\Security\User;

class DeepLApi
{
	protected EntityManagerDecorator $em;
	protected User                   $user;

	protected string $authKey;
	protected array  $locales = [
		'cs' => 'cs',
		'en' => 'en-GB',
		'es' => 'es',
		'ua' => 'uk',
	];

	public function __construct(
		string                 $authKey,
		EntityManagerDecorator $em,
		User                   $user
	)
	{
		$this->authKey = $authKey;
		$this->em      = $em;
		$this->user    = $user;
	}

	public function translateString(string $str, string $sourceLang, string $targetLang): string
	{
		if (Validators::isUrl($str) || Validators::isUri($str)) {
			return $str;
		}

		if ($str === strip_tags($str)) {
			$result = $this->translateText($str, $sourceLang, $targetLang);
		} else {
			$result = $this->translateHtml($str, $sourceLang, $targetLang);
		}

		return is_array($result) ? (string) reset($result) : $result;
	}

	/**
	 * @param string|string[] $data
	 *
	 * @return string|array
	 */
	public function translateText($data, string $sourceLang, string $targetLang)
	{
		if ($data === '' || !in_array($targetLang, (array) Parameters::load('deepL.whitelist') ?: [], true)) {
			return $data;
		}

		$translator = new Translator($this->authKey);

		$resultApi = $translator->translateText($data, $this->getTargetLocale($sourceLang), $this->getTargetLocale($targetLang), [
			'split_sentences'        => 1,
			'preserve_formatting'    => true,
			'show_billed_characters' => true,
		]);

		if (is_array($resultApi)) {
			$result = [];

			foreach ($resultApi as $k => $resultText) {
				$result[] = $resultText->text;

				if ($resultText->billedCharacters) {
					$this->logTranslate($sourceLang, $targetLang, (string) $data[$k], $resultText->text, (int) $resultText->billedCharacters);
				}
			}

			return $result;
		}

		if ($resultApi->billedCharacters) {
			$this->logTranslate($sourceLang, $targetLang, is_array($data) ? (string) array_values($data)[0] : (string) $data, $resultApi->text, (int) $resultApi->billedCharacters);
		}

		return $resultApi->text;
	}

	/**
	 * @param string|string[] $data
	 *
	 * @return string|array
	 */
	public function translateHtml($data, string $sourceLang, string $targetLang)
	{
		if ($data === '' || (is_string($data) && strip_tags($data) === '') || !in_array($targetLang, (array) Parameters::load('deepL.whitelist') ?: [], true)) {
			return $data;
		}

		$translator = new Translator($this->authKey);

		if (is_array($data)) {
			$dataOrigin = [];

			foreach ($data as $k => $v) {
				$dataOrigin[$k] = $v;
				$data[$k]       = str_replace("\n", '||||', $v);
			}
		} else {
			$dataOrigin = $data;
			$data       = str_replace("\n", '||||', $data);
		}

		$resultApi = $translator->translateText($data, $this->getTargetLocale($sourceLang), $this->getTargetLocale($targetLang), [
			'tag_handling'           => 'html',
			'show_billed_characters' => true,
		]);

		if (is_array($resultApi)) {
			$result = [];

			foreach ($resultApi as $k => $v) {
				$result[$k] = $v->text;

				if ($v->billedCharacters) {
					$this->logTranslate($sourceLang, $targetLang, (string) $dataOrigin[$k], $result[$k], (int) $v->billedCharacters);
				}

				$result[$k] = str_replace('||||', "\n", $result[$k]);
			}
		} else {
			$result = str_replace('||||', "\n", $resultApi->text);

			if ($resultApi->billedCharacters) {
				$this->logTranslate($sourceLang, $targetLang, is_array($dataOrigin) ? (string) array_values($dataOrigin)[0] : (string) $dataOrigin, $result, (int) $resultApi->billedCharacters);
			}
		}

		return $result;
	}

	public function getUsage(): Usage
	{
		$translator = new Translator($this->authKey);

		return $translator->getUsage();
	}

	protected function getTargetLocale(string $lang): string
	{
		return $this->locales[$lang] ?? $lang;
	}

	protected function logTranslate(string $sourceLang, string $targetLang, string $sourceText, string $targetText, int $billedCharacters): void
	{
		/** @var ?\Users\Model\Entities\User $user */
		$user = $this->user->isLoggedIn() ? $this->user->getIdentity() : null;

		try {

			$this->em->getConnection()->insert('core__translate_log', [
				'source_locale'     => $sourceLang,
				'target_locale'     => $targetLang,
				'source_text'       => $sourceText,
				'target_text'       => $targetText,
				'user'              => $user ? $user->getId() : null,
				'user_email'        => $user ? $user->email : null,
				'billed_characters' => $billedCharacters,
			]);
		} catch (Exception $e) {
			Debugger::log($e->getMessage(), 'translate_log');
			Debugger::log([
				'source_locale'     => $sourceLang,
				'target_locale'     => $targetLang,
				'source_text'       => $sourceText,
				'target_text'       => $targetText,
				'user'              => $user ? $user->getId() : null,
				'user_email'        => $user ? $user->email : null,
				'billed_characters' => $billedCharacters,
			], 'translate_log');
		}
	}
}
