<?php declare(strict_types = 1);

namespace Core\Model\Turnstile;

use Core\Model\SystemConfig;
use Nette\Forms\Controls\BaseControl;
use Tracy\Debugger;

class Turnstile
{
	protected const apiEndpoint    = 'https://challenges.cloudflare.com/turnstile/v0/';
	protected const defaultTimeout = 15;

	protected string $secretKey;
	protected int    $timeout = self::defaultTimeout;

	public function __construct(
		string $secretKey,
		int    $timeout = self::defaultTimeout
	)
	{
		$this->setSecretKey($secretKey);
		$this->setTimeout($timeout);
	}

	public function setSecretKey(string $secretKey): void
	{
		$this->secretKey = $secretKey;
	}

	public function setTimeout(int $seconds): void
	{
		$this->timeout = $seconds;
	}

	public function verify(
		string  $response,
		?string $remoteIp = null
	): ?Response
	{
		$curl = curl_init();

		curl_setopt_array($curl, [
			CURLOPT_URL            => self::apiEndpoint . 'siteverify',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_TIMEOUT        => $this->timeout,
			CURLOPT_POST           => true,
			CURLOPT_POSTFIELDS     => http_build_query([
				'secret'   => $this->secretKey,
				'response' => $response,
				'remoteip' => $remoteIp,
			]),
		]);

		$response = curl_exec($curl);

		curl_close($curl);

		if (SystemConfig::load('turnstile.allowLogResponse')) {
			Debugger::log('Turnstile response: ' . $response, 'turnstileResponse');
		}

		return is_string($response) ? Response::deserialize($response) : null;
	}

	public function validateControl(BaseControl $control): bool
	{
		$response = $this->verify($control->getValue(), $_SERVER['REMOTE_ADDR'] ?? null);

		return $response && $response->isSuccess();
	}
}
