<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid;

use Contributte\Datagrid\Column\Column;
use Contributte\Datagrid\Column\ColumnText;
use Contributte\Datagrid\Components\DatagridPaginator\DatagridPaginator;
use Contributte\Datagrid\Datagrid;
use Contributte\Datagrid\DataModel;
use Contributte\Datagrid\DataSource\IDataSource;
use Contributte\Datagrid\Filter\SubmitButton;
use Contributte\Datagrid\GroupAction\GroupAction;
use Contributte\Datagrid\InlineEdit\InlineAdd;
use Contributte\Datagrid\InlineEdit\InlineEdit;
use Core\Model\Lang\Langs;
use Core\Model\Translation\FieldLockerService;
use Core\Model\UI\DataGrid\Column\Action;
use Core\Model\UI\DataGrid\Column\ColumnPrice;
use Core\Model\UI\DataGrid\Column\ColumnStatus;
use Core\Model\UI\DataGrid\Filter\FilterCheckboxNestedList;
use Core\Model\UI\DataGrid\Filter\FilterRadioNestedList;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\UI\Form;
use Nette\ComponentModel\IContainer;
use Nette\Forms\Container;
use Nette\Forms\Form as NetteForm;
use Users\Model\Security\User;

class BaseDataGrid extends Datagrid
{
	protected string $tableClass      = '';
	protected string $sortableGroup   = '';
	protected bool   $hideTableFooter = false;

	public ?Langs              $langsService       = null;
	public ?User               $user               = null;
	public ?FieldLockerService $fieldLockerService = null;

	public string $detailEyePosition = 'right';

	protected ?CustomGroupActionCollection $groupActionCustomCollection = null;
	protected array                        $headFilters                 = [];
	protected array                        $customData                  = [];
	public bool                            $disableFilterFormValidation = false;

	public function __construct(IContainer $parent = null, ?string $name = null)
	{
		parent::__construct($parent, $name);

		$this->setDefaultPerPage(20);
		$this->setRefreshUrl(false);
		$this->setRememberState(true);
		$this->setOuterFilterColumnsCount(4);

		$this->setTemplateFile(__DIR__ . '/templates/datagrid.latte');

		$this->onAnchor[] = function() {
			$this->getTemplate()->newTemplate = __DIR__ . '/templates/datagrid.latte';
		};
	}

	public function render(): void
	{
		if ($this->isTreeView()) {
			$this->template->extendTemplate = ROOT_DIR . '/vendor/ublaboo/datagrid/src/templates/datagrid_tree.latte';
			$this->setTemplateFile(__DIR__ . '/templates/datagrid_tree.latte');
		} else {
			$this->template->extendTemplate = $this->getOriginalTemplateFile();
		}

		$this->template->tableClass      = $this->tableClass;
		$this->template->sortableGroup   = $this->sortableGroup;
		$this->template->hideTableFooter = $this->hideTableFooter;
		$this->template->headFilters     = $this->headFilters;

		parent::render();
	}

	/**
	 * @param array|array|int[]|array|string[] $itemsPerPageList
	 *
	 * @return static
	 */
	public function setItemsPerPageList(array $itemsPerPageList, bool $includeAll = true): DataGrid
	{
		$this->itemsPerPageList = $itemsPerPageList;

		if ($includeAll) {
			$this->itemsPerPageList[] = 'all';
		}

		if (count($this->getItemsPerPageList()) === 1) {
			$this->setPagination(false);
		}

		return $this;
	}

	public function addAction(string $key, string $name, ?string $href = null, array $params = null): Action
	{
		$this->addActionCheck($key);

		$href = $href ?: $key;

		if ($params === null) {
			$params = [$this->primaryKey];
		}

		$control = $this->actions[$key] = new Action($this, $key, $href, $name, $params);

		return $control;
	}

	protected function customFilterComponent(): BaseForm
	{
		$form = new BaseForm($this, 'filter');
		$form->setLangsService($this->langsService);
		$form->setFieldLockerService($this->fieldLockerService);
		$form->setUser($this->user);
		$form->setTranslator($this->getTranslator());
		$form->setMethod(static::$formMethod);

		if ($this->disableFilterFormValidation) {
			$form->setHtmlAttribute('formnovalidate', 'true');
		}


		/**
		 * InlineEdit part
		 */
		$inline_edit_container = $form->addContainer('inline_edit');

		if ($this->inlineEdit instanceof InlineEdit) {
			$inline_edit_container->addSubmit('submit', 'contributte_datagrid.save')
				->setValidationScope([$inline_edit_container]);
			$inline_edit_container->addSubmit('cancel', 'contributte_datagrid.cancel')
				->setValidationScope(null);

			$this->inlineEdit->onControlAdd($inline_edit_container);
			$this->inlineEdit->onControlAfterAdd($inline_edit_container);
		}

		/**
		 * InlineAdd part
		 */
		$inlineAddContainer = $form->addContainer('inline_add');

		if ($this->inlineAdd instanceof InlineAdd) {
			$inlineAddContainer->addSubmit('submit', 'contributte_datagrid.save')
				->setValidationScope([$inlineAddContainer]);
			$inlineAddContainer->addSubmit('cancel', 'contributte_datagrid.cancel')
				->setValidationScope(null)
				->setAttribute('data-datagrid-cancel-inline-add', true);

			$this->inlineAdd->onControlAdd($inlineAddContainer);
			$this->inlineAdd->onControlAfterAdd($inlineAddContainer);
		}

		/**
		 * ItemDetail form part
		 */
		$itemsDetailForm = $this->getItemDetailForm();

		if ($itemsDetailForm instanceof Container) {
			$form['items_detail_form'] = $itemsDetailForm;
		}

		/**
		 * Filter part
		 */
		$filterContainer = $form->addContainer('filter');

		foreach ($this->filters as $filter) {
			$filter->addToFormContainer($filterContainer);
		}

		if (!$this->hasAutoSubmit()) {
			$filterContainer['submit'] = $this->getFilterSubmitButton();
		}

		/**
		 * Group action part
		 */
		$groupActionContainer = $form->addContainer('group_action');

		if ($this->hasGroupActions()) {
			$this->getGroupActionCollection()->addToFormContainer($groupActionContainer);
		}

		$groupActionCustomContainer = $form->addContainer('groupActionCustom');

		$this->getGroupActionCustomCollection()->addToFormContainer($groupActionCustomContainer);

		if ($form->isSubmitted() === false) {
			$this->setFilterContainerDefaults($filterContainer, $this->filter);
		}

		/**
		 * Per page part
		 */
		$select = $form->addSelect('perPage', '', $this->getItemsPerPageList())
			->setTranslator(null);

		if ($form->isSubmitted() === false) {
			$select->setValue($this->getPerPage());
		}

		$form->addSubmit('perPage_submit', 'contributte_datagrid.perPage_submit')
			->setValidationScope([$select]);

		$form->onSubmit[] = $this->filterSucceeded(...);

		return $form;
	}

	public function filterSucceeded(NetteForm $form): void
	{
		parent::filterSucceeded($form);

		try {
			$sort = $this->sort ?: $this->defaultSort;
			if (is_null($this->sortCallback) && $sort) {
				$column             = $this->getColumn(key($sort));
				$this->sortCallback = $column->getSortableCallback();
			}
		} catch (Exception) {
		}
	}

	public function createComponentFilter(): Form
	{
		return $this->customFilterComponent();
	}

	public function getDataModel(): ?DataModel
	{
		return $this->dataModel;
	}

	public function getDataSource(): array|IDataSource|null
	{
		return parent::getDataSource();
	}

	public function handleResetFilter(): void
	{
		parent::handleResetFilter();
	}

	public function handlePage(int $page): void
	{
		try {
			$sort = $this->sort ?: $this->defaultSort;
			if (is_null($this->sortCallback) && $sort) {
				$column             = $this->getColumn(key($sort));
				$this->sortCallback = $column->getSortableCallback();
			}
		} catch (Exception) {
		}

		parent::handlePage($page);
	}

	public function createComponentPaginator(): DatagridPaginator
	{
		$control = parent::createComponentPaginator();

		$control->onAnchor[] = function() use ($control) {
			$control->template->originTemplate = $control->getTemplateFile();

			$control->setTemplateFile(__DIR__ . '/templates/data_grid_paginator.latte');
		};

		return $control;
	}

	public function addColumnStatus(string $key, string $name, ?string $column = null): ColumnStatus
	{

		$column ??= $key;

		$columnStatus = new ColumnStatus($this, $key, $column, $name);
		$this->addColumn($key, $columnStatus);

		return $columnStatus;
	}

	public function addColumnEmail(string $key, string $name, ?string $column = null): ColumnText
	{
		$column = $this->addColumnText($key, $name, $column);
		$column->setTemplate(__DIR__ . '/templates/column_email.latte', [
			'key' => $column->getColumn(),
		]);

		return $column;
	}

	public function addColumnPhone(
		string  $key,
		string  $name,
		?string $column = null,
		?string $country = null,
		bool    $showRegion = false,
	): ColumnText
	{
		$column = $this->addColumnText($key, $name, $column);

		$column->setAlign('right');
		$column->setTemplate(__DIR__ . '/templates/column_phone.latte', [
			'key'        => $column->getColumn(),
			'keyRegion'  => $country,
			'showRegion' => $showRegion,
		]);

		return $column;
	}

	public function addColumnPrice(string $key, string $name, ?string $column = null): Column
	{
		$column = $column ?: $key;

		return $this->addColumn($key, new ColumnPrice($this, $key, $column, $name));
	}

	public function addFilterRadioNestedList(string $key, string $name, array $flatItems): FilterRadioNestedList
	{
		$this->addFilterCheck($key);

		return $this->filters[$key] = new FilterRadioNestedList($this, $key, $name, $flatItems);
	}

	public function addFilterCheckboxNestedList(
		string $key,
		string $name,
		array  $flatItems,
	): FilterCheckboxNestedList
	{
		$this->addFilterCheck($key);

		return $this->filters[$key] = new FilterCheckboxNestedList($this, $key, $name, $flatItems);
	}

	public function setHeadFilters(array $items = []): self
	{
		$this->headFilters = $items;

		return $this;
	}

	public function addGroupSelectCustomAction(string $title, array $options = []): GroupAction
	{
		return $this->getGroupActionCustomCollection()->addGroupSelectAction($title, $options);
	}

	public function setTableClass(string $class): self
	{
		$this->tableClass = $class;

		return $this;
	}

	public function setSortableGroup(string $group): self
	{
		$this->sortableGroup = $group;

		return $this;
	}

	public function hideTableFooter(bool $hide = true): self
	{
		$this->hideTableFooter = $hide;

		return $this;
	}

	public function getGroupActionCustomCollection(): CustomGroupActionCollection
	{
		if ($this->groupActionCustomCollection === null) {
			$this->groupActionCustomCollection = new CustomGroupActionCollection($this);
		}

		return $this->groupActionCustomCollection;
	}

	/**
	 * @param mixed|null $value
	 */
	public function addCustomData(string $key, $value): self
	{
		$this->customData[$key] = $value;

		return $this;
	}

	/**
	 * @return mixed|null
	 */
	public function getCustomData(string $key) { return $this->customData[$key] ?? null; }

	public function getFilterSubmitButton(): SubmitButton
	{
		$button = parent::getFilterSubmitButton();

		$button->setText('');

		return $button;
	}

}
