<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid\Column;

use Contributte\Datagrid\Column\Action\Confirmation\CallbackConfirmation;
use Contributte\Datagrid\Column\Action\Confirmation\IConfirmation;
use Contributte\Datagrid\Column\Action\Confirmation\StringConfirmation;
use Contributte\Datagrid\Datagrid;
use Contributte\Datagrid\Exception\DatagridColumnRendererException;
use Contributte\Datagrid\Row;
use Core\Model\UI\DataGrid\Traits\TButtonTryAddIcon;
use Exception;
use Nette\Utils\Html;
use Ublaboo;

class Action extends \Contributte\Datagrid\Column\Action
{
	use TButtonTryAddIcon;

	/** @var string */
	protected        $class  = 'btn btn-xs';
	protected string $bsType = 'secondary';

	public function render(Row $row): mixed
	{
		if (!$this->shouldBeRendered($row)) {
			return null;
		}

		try {
			return $this->useRenderer($row);
		} catch (DatagridColumnRendererException) {
		}

		$link = $this->createLink(
			$this->grid,
			$this->href,
			$this->getItemParams($row, $this->params) + $this->parameters,
		);

		$a = Html::el('a')->href($link);

		$this->tryAddIcon($a, $this->getIcon($row), $this->getName());

		if ($this->dataAttributes !== []) {
			foreach ($this->dataAttributes as $key => $value) {
				$a->data((string) $key, $value);
			}
		}

		if ($this->attributes !== []) {
			$a->addAttributes($this->attributes);
		}

		if ($this->getName()) {
			$a->addHtml(Html::el('span')->setText($this->translate($this->getName())));
		}

		$title = $this->getTitle($row);

		if ($title !== null) {
			$a->setAttribute('title', $this->translate($title));
		}

		if ($this->class !== null) {
			$a->setAttribute('class', $this->getClass($row));
		}

		$confirmationDialog = $this->getConfirmationDialog($row);

		if ($confirmationDialog !== null) {
			$a->data(static::$dataConfirmAttributeName, $confirmationDialog);
		}

		if ($this->openInNewTab) {
			$a->addAttributes(['target' => '_blank']);
		}

		/** @phpstan-ignore-next-line  */
		if ($a instanceof Html) {
			$a->class .= ' btn-' . $this->bsType;
		}

		return $a;
	}

	public function addClass(string $class): self
	{
		$this->class .= ' ' . $class;

		return $this;
	}

	public function setBsType(string $type = 'secondary'): self
	{
		$this->bsType = $type;

		return $this;
	}

	public function setConfirm(string|callable $val): self
	{
		if ($val instanceof IConfirmation) {
			$this->setConfirmation($val);

			return $this;
		}

		$confirm = null;
		if (is_string($val)) {
			$confirm = new StringConfirmation($val);
		}

		if (is_callable($val)) {
			$confirm = new CallbackConfirmation($val);
		}

		if ($confirm) {
			$this->setConfirmation($confirm);
		}

		return $this;
	}

	protected function createLink(DataGrid $grid, string $href, array $params): string
	{
		try {
			$link = parent::createLink($grid, $href, $params);
		} catch (Exception) {
			$link = $this->grid->getPresenter()->link($href, $params);
		}

		return $link;
	}

}
