<?php declare(strict_types = 1);

namespace Core\Model\UI\Form;

use Core\Model\Helpers\CoreHelper;
use Core\Model\Lang\Langs;
use Core\Model\Module;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Core\Model\SystemConfig;
use Core\Model\Translation\FieldLockerService;
use Core\Model\Turnstile\Turnstile;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\Form\Controls\TurnstileInput;
use Core\Model\UI\Form\Traits\BaseContainerTrait;
use Core\Model\UI\Form\Traits\BootstrapContainerTrait;
use Nette\Application\UI\Form;
use Nette\ComponentModel\Component;
use Nette\ComponentModel\IContainer;
use Nette\Forms\FormRenderer;
use Nette\InvalidArgumentException;
use Nette\Utils\Html;
use Nextras;
use Tracy\Debugger;
use Users\Model\Security\User;

/**
 * @method AbstractPresenter getPresenter()
 * @method AbstractPresenter getPresenterIfExists()
 *
 * @property-read AbstractPresenter $presenter
 * @property-read BootstrapRenderer $renderer
 */
class BaseForm extends Form
{
	use BootstrapContainerTrait;
	use BaseContainerTrait;

	/** @var string|Html */
	public $description;

	/** @var callable[] */
	public $onSuccessSave;

	/** @var callable[] */
	public $onSuccessSaveAndClose;

	/** @var callable[] */
	public $onCancel;

	protected bool             $disableSuccessSave  = false;
	protected bool             $isAjax              = true;
	protected array            $customData          = [];
	public ?Langs              $langsService        = null;
	public ?User               $user                = null;
	public ?FieldLockerService $fieldLockerService  = null;
	protected bool             $showLangSwitcher    = true;
	protected bool             $globalXSSProtection = true;

	public static string $deeplTranslatorBlock = __DIR__ . '/DeepLTranslatorBlock.latte';

	protected array $headLinks = [];

	/**
	 * @param IContainer|null $container
	 *
	 */
	public function __construct($container = null, ?string $name = null)
	{
		parent::__construct($container, $name);
		$this->setRenderer(new BootstrapRenderer);

		if (Module::isFront()) {
			$this->onValidate[] = [$this, 'validateXSSGlobally'];
		}
	}

	protected function beforeRender(): void
	{
		parent::beforeRender();
		foreach ($this->getControls() as $control) {
			if ($control->isRequired()) {
				$control->getLabelPrototype()->class('required', true);
			}
		}
	}

	public function getRenderer(): FormRenderer
	{
		/** @var BootstrapRenderer $renderer */
		$renderer = parent::getRenderer();
		$renderer->setForm($this);

		return $renderer;
	}

	public function setRenderer(FormRenderer $renderer = null): static
	{
		if (!$renderer instanceof BootstrapRenderer) {
			throw new InvalidArgumentException('Must be a BootstrapRenderer');
		}

		return parent::setRenderer($renderer);
	}

	public function setGlobalXSSProtection(bool $enable = true): self
	{
		$this->globalXSSProtection = $enable;

		return $this;
	}

	public function validateXSSGlobally(Form $form): void
	{
		if (!$this->globalXSSProtection) {
			return;
		}

		$values = $form->getValues('array');
		$this->checkValuesForXSS($values);
	}

	/**
	 * @param mixed $values
	 */
	protected function checkValuesForXSS($values, string $path = ''): void
	{
		if (is_array($values)) {
			foreach ($values as $key => $value) {
				$currentPath = $path ? $path . '.' . $key : $key;

				if (is_string($value)) {
					$this->validateXSSString($value, $currentPath);
				} else if (is_array($value)) {
					$this->checkValuesForXSS($value, $currentPath);
				}
			}
		} else if (is_string($values)) {
			$this->validateXSSString($values, $path);
		}
	}

	protected function validateXSSString(string $value, string $fieldPath): void
	{
		$dangerous = [
			'<script', '</script>', 'javascript:', 'onload=', 'onerror=',
			'onclick=', 'onmouseover=', 'onfocus=', 'onblur=', '<iframe',
			'eval(', 'document.cookie', 'window.location', 'data:text/html',
			'vbscript:', '<meta', '<object', '<embed', '<link', 'expression(',
		];

		$decodedVariants = [
			$value,
			urldecode($value),
			html_entity_decode($value, ENT_QUOTES | ENT_HTML5, 'UTF-8'),
			base64_decode($value, true) ?: '',
		];

		foreach ($decodedVariants as $variant) {
			$lowerValue = strtolower($variant);

			foreach ($dangerous as $pattern) {
				if (strpos($lowerValue, $pattern) !== false) {
					$data = [
						'type'       => 'XSS_ATTEMPT',
						'input'      => $value,
						'field'      => $fieldPath,
						'url'        => CoreHelper::getCurrentUrl(),
						'form'       => get_class($this),
						'ip'         => CoreHelper::getUserIpAddress(),
						'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
						'timestamp'  => date('Y-m-d H:i:s'),
					];
					Debugger::log(json_encode($data), 'xss-security');

					$jsonStringPretty = (string) json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

					LogNotifier::toDevelopers(htmlspecialchars($jsonStringPretty, ENT_QUOTES, 'UTF-8'), 'XSS attempt detected.', true);

					die('XSS attempt detected. Request terminated.');
				}
			}
		}
	}

	public function isAjax(): bool
	{
		return $this->isAjax;
	}

	public function setAjax($isAjax = true): static
	{
		$this->isAjax = $isAjax;

		if ($isAjax) {
			$this->getForm()->getElementPrototype()->addClass('ajax');
		}

		return $this;
	}

	public function setRenderMode(int $renderMode): void
	{
		$this->renderer->setMode($renderMode);
	}

	public function disableSuccessSave(bool $disable = true): void { $this->disableSuccessSave = $disable; }

	public function fireEvents(): void
	{
		if ($this->isSubmitted() instanceof Component && $this->isSubmitted()->name === 'cancel') {
			$this->onCancel($this);

			return;
		}
		parent::fireEvents();

		if ($this->isValid() && !$this->disableSuccessSave) {
			$submittedBy = $this->isSubmitted() instanceof Component ? $this->isSubmitted()->name : null;

			if ($submittedBy == 'save') {
				$this->onSuccessSave($this);
			} else if ($submittedBy == 'saveAndClose') {
				$this->onSuccessSaveAndClose($this);
			}
		}
	}

	/**
	 * @param array $arr
	 *
	 * @return $this
	 */
	public function setCustomData($arr)
	{
		$this->customData = $arr;

		return $this;
	}

	/**
	 * @param string $key
	 *
	 * @return $this
	 */
	public function addCustomData($key, mixed $value)
	{
		$this->customData[$key] = $value;

		return $this;
	}

	public function addTurnstile(
		string  $name,
		string  $label,
		?string $message = null,
		?string $secretKey = null,
		?string $publicKey = null
	): TurnstileInput
	{
		$field = new TurnstileInput(
			new Turnstile($secretKey ?: (string) SystemConfig::loadScalar('turnstile.secretKey')),
			$label,
			$message,
			$publicKey ?: (string) SystemConfig::loadScalar('turnstile.publicKey')
		);
		$field->setRequired();

		$this->addComponent($field, $name);

		return $field;
	}

	/**
	 *
	 * @return mixed
	 */
	public function getCustomData(mixed $key, mixed $default = null) { return $this->customData[$key] ?: $default; }

	public function setLangsService(Langs $service): self
	{
		$this->langsService = $service;

		return $this;
	}

	public function setUser(User $user): self
	{
		$this->user = $user;

		return $this;
	}

	public function setFieldLockerService(FieldLockerService $fieldLockerService): self
	{
		$this->fieldLockerService = $fieldLockerService;

		return $this;
	}

	public function getDescription(): ?Html
	{
		if (!$this->description)
			return null;

		return $this->description instanceof Html ? $this->description : Html::el()->setHtml($this->description);
	}

	/**
	 * @return $this
	 */
	public function setDescription(string|Html $description): self
	{
		$this->description = $description;

		return $this;
	}

	/**
	 * @param array $inputs
	 * @param array $values
	 *
	 * @return array
	 */
	public function prepareMultilangTextsForEntity($inputs, $values)
	{
		$mTexts = [];
		$langs  = array_keys($this->langsService->getLangs(false));
		foreach ($inputs as $k => $v) {
			if (is_numeric($k))
				$k = $v;

			foreach ($values->$k as $lang => $val) {
				if (in_array($lang, $langs)) {
					$mTexts[$lang][$v] = $val;
				}
			}
		}

		return $mTexts;
	}

	/**
	 * @param array $inputs
	 * @param array $texts
	 *
	 * @return array
	 */
	public function prepareMultilangTextsForForm($inputs, $texts)
	{
		$mTexts  = [];
		$seoData = [];

		foreach ($texts as $lang => $text) {
			foreach ($inputs as $k => $v) {
				if (is_numeric($k))
					$k = $v;

				$mTexts[$k][$lang] = $text->$v;
			}

			$class = $text::class;
			if ($class && property_exists($class, 'seo')) {
				foreach ($text->getSeo() as $k => $v) {
					$seoData[$k][$lang] = $v;
				}
			}
		}

		return ['texts' => $mTexts, 'seo' => $seoData];
	}

	public function convertMultilangValuesToArray(): array
	{
		$result = [];
		if ($this->isAjax()) {
			$values = $this->getHttpData();
		} else {
			$values = $this->getValues('array');
		}

		foreach ($this->getComponents(true) as $control) {
			$name = $control->getName();

			if (in_array($name, ['saveControl']) || $control instanceof BaseContainer) {
				continue;
			}

			if (isset($control->isMultiLanguage) && $control->isMultiLanguage == true) {
				$parents = [];
				$con     = $control;
				while ($con->getParent() instanceof BaseContainer) {
					$con       = $con->getParent();
					$parents[] = $con->getName();
				}

				$parents = array_reverse($parents);
				$val     = $values;

				foreach ($parents as $parent) {
					$val = $val[$parent];
				}

				foreach ($val[$name] as $l => $v) {
					if (!array_key_exists($l, $result)) {
						$result[$l] = [];
					}

					/** @var array $target */
					$target = &$result[$l];
					foreach ($parents as $parent) {
						if (!array_key_exists($parent, $target)) {
							$target[$parent] = [];
						}

						$target = &$target[$parent];
					}

					$target[$name] = $v;
				}
			}
		}

		return $result;
	}

	public function getShowLangSwitcher(): bool { return $this->showLangSwitcher; }

	public function setShowLangSwitcher(bool $show = true): self
	{
		$this->showLangSwitcher = $show;

		return $this;
	}

	public function getHeadLinks(): array { return $this->headLinks; }

	public function addHeadLink(string $link, string $text, ?string $class = null, ?string $baseRowClass = null): self
	{
		$this->headLinks[] = [
			'link'         => $link,
			'text'         => $text,
			'class'        => $class,
			'baseRowClass' => $baseRowClass,
		];

		return $this;
	}

	public function addHeadLinkEl(Html $el, ?string $baseRowClass = null): self
	{
		$this->headLinks[] = [
			'el'           => $el,
			'baseRowClass' => $baseRowClass,
		];

		return $this;
	}
}
