<?php declare(strict_types = 1);

namespace Core\Model\UI\Form;

use Nette\Forms\IFormRenderer;
use Nette\SmartObject;
use Exception;
use Nette\Forms\Container;
use Nette\InvalidArgumentException;
use Nette\Forms\Controls\TextInput;
use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\Controls\BoolInput;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\SaveCancelControl;
use Core\Model\UI\Form\Controls\SelectInput;
use Core\Model\UI\Form\Controls\UploadInput;
use Core\Model\UI\Form\Enums\RenderMode;
use Nette;
use Nette\ComponentModel\IComponent;
use Nette\Forms\ControlGroup;
use Nette\Forms\Controls\BaseControl;
use Nette\Forms\Form;
use Nette\Forms\IControl;
use Nette\Utils\Html;

/**
 * @property-read BaseForm $form
 * @property int           $mode
 */
class BootstrapRenderer implements IFormRenderer
{
	use SmartObject;

	protected const defaultLabelColumns   = 3;
	protected const defaultControlColumns = 9;

	/**
	 * Možno zvolit přímo šablonu nebo levou a pravou stranu
	 *
	 * $extendedLayout = [
	 *      'tabTitle' => [
	 *          'file' => path
	 *          'left' => [controlName],
	 *          'right' => [controlName],
	 *      ],
	 * ]
	 * @var array
	 */
	public $extendedLayout = [];

	final public const BASE_EXTENDED_NAME = 'default.extendedSettings';

	/**
	 * @var array of HTML tags
	 */
	public $wrappers = [
		'form' => [
			'container' => null,
		],

		'error' => [
			'container'       => 'div class="frm__errors"',
			'item'            => 'div class="alert alert-danger" role=alert',
			'dismiss'         => 'button class=close type=button data-dismiss=alert aria-label=Close',
			'dismiss_in'      => 'span aria-hidden=true',
			'dismiss_content' => '&times;',
		],

		'group' => [
			'container'   => 'fieldset',
			'label'       => 'legend',
			'description' => 'p',
		],

		'pair' => [
			'container'        => 'div class=frm__pair',
			'side_container'   => 'div class="form-group row"',
			'inline_container' => 'div class="form-group mr-2 mb-2 mt-2"',
			'.error'           => 'has-danger',
			'.required'        => 'required',
		],

		'control' => [
			'container'      => null,
			'side_container' => "div class=col-sm-" . self::defaultControlColumns,
			'save_cancel'    => 'div class=btn-group',

			'description'    => 'div class="frm__input-description"',
			'errorcontainer' => 'div class=form-control-feedback',
			'erroritem'      => 'div class="frm__input-error"',

			'.file'   => 'div',
			'.select' => 'div',
			'.button' => 'button',
		],

		'label' => [
			'container'        => null,
			'side_container'   => null,
			'inline_container' => 'div class=mr-2',

			'side_class' => 'col-form-label text-md-right col-sm-' . self::defaultLabelColumns,
		],

		'hidden' => [
			'container' => null,
		],
	];

	/** @var BaseForm|null */
	protected ?Nette\Application\UI\Form $form = null;

	protected int  $counter        = 0;
	protected int  $labelColumns   = self::defaultLabelColumns;
	protected int  $controlColumns = self::defaultControlColumns;
	protected int  $renderMode     = RenderMode::SideBySideMode;
	protected bool $autocomplete   = true;

	public function __construct(int $mode = RenderMode::VerticalMode)
	{
		$this->setMode($mode);
	}

	public function setMode(int $renderMode): self
	{
		$this->renderMode = $renderMode;

		return $this;
	}

	public function getAutocomplete(): bool { return $this->autocomplete; }

	public function setAutocomplete(bool $value): self
	{
		$this->autocomplete = $value;

		return $this;
	}

	/**
	 * Returns render mode
	 * @return int
	 * @see RenderMode
	 */
	public function getMode()
	{
		return $this->renderMode;
	}

	public function setForm(BaseForm $form): void { $this->form = $form; }

	function render(Form $form): string
	{
		if ($this->form !== $form && $form instanceof BaseForm) {
			$this->form = $form;
		}

		$s = '';
		$s .= $this->renderBegin();
		$s .= $this->renderErrors();
		$s .= $this->renderErrors(null, false);
		$s .= $this->renderBody();
		$s .= $this->renderEnd();

		return $s;
	}

	/**
	 * Renders form begin.
	 * @return string
	 */
	public function renderBegin()
	{
		$this->counter = 0;
		$output        = '';

		foreach ($this->form->getControls() as $control) {
			$control->setOption('rendered', false);
		}

		$prototype = clone $this->form->getElementPrototype();
		if ($this->mode === RenderMode::Inline) {
			$prototype->addClass('form-inline');
		}
		if ($this->form->isAjax()) {
			$prototype->addClass('ajax');
		}

		if ($this->form->isMethod('get')) {
			$el         = $prototype;
			$query      = parse_url($el->action, PHP_URL_QUERY);
			$el->action = str_replace("?$query", '', $el->action);
			$s          = '';
			foreach (preg_split('#[;&]#', (string) $query, -1, PREG_SPLIT_NO_EMPTY) ?: [] as $param) {
				$parts = explode('=', $param, 2);
				$name  = urldecode($parts[0]);
				if (!isset($this->form[$name])) {
					$s .= Html::el('input', ['type' => 'hidden', 'name' => $name, 'value' => urldecode($parts[1])]);
				}
			}

			$output = $el->startTag() . ($s ? "\n\t" . $this->getWrapper('hidden container')->setHtml($s) : '');
		} else {
			$output = $prototype->startTag();
		}

		$output .= (string) $this->renderHead($this->form);

		if ($this->form->getDescription()) {
			$output .= (string) (Html::el('div class=frm__description'))->addHtml($this->form->getDescription());
		}

		return $output;
	}

	protected function getWrapper(string $name): Html
	{
		$data = $this->getValue($name);

		return $data instanceof Html ? clone $data : Html::el($data);
	}

	/**
	 * @return string
	 */
	protected function getValue(string $name)
	{
		$name = explode(' ', $name);
		$data = &$this->wrappers[$name[0]][$name[1]];

		return $data;
	}

	public function renderHead(BaseForm $form): Html
	{
		$formHead = Html::el('div class=frm__base-head-row');

		$baseRowItem = Html::el('div class=frm__base-head-row__item');
		if (isset($form['saveControl'])) {
			$saveControlRowItem = clone $baseRowItem;
			$saveControlRowItem->addHtml($this->renderControls($form['saveControl']));
			$formHead->addHtml($saveControlRowItem);
		}

		if ($form->getHeadLinks()) {
			foreach ($form->getHeadLinks() as $link) {
				$tmp = clone $baseRowItem;
				if ($link['baseRowClass'])
					$tmp->addClass($link['baseRowClass']);

				if (isset($link['el']))
					$tmp->addHtml($link['el']);
				else {
					$tmp->addHtml(
						Html::el('a', [
							'class' => 'btn ' . $link['class'],
							'href'  => $link['link'],
						])->setText($link['text']),
					);
				}

				$formHead->addHtml($tmp);
			}
		}

		if ($form->getComponent('isPublished', false)) {
			$formHead->addHtml(
				(clone $baseRowItem)->addHtml($this->renderPair($form->getComponent('isPublished'))),
			);
		}

		if ($form->getComponent('isActive', false)) {
			$formHead->addHtml(
				(clone $baseRowItem)->addHtml($this->renderPair($form->getComponent('isActive'))),
			);
		}

		try {
			if ($form->getParent()->getComponent('contentLangSwitcher') && $form->getShowLangSwitcher()) {
				ob_start();
				/** @phpstan-ignore-next-line */
				$form->getParent()['contentLangSwitcher']->render();
				$tmp = ob_get_clean();
				$formHead->addHtml((clone $baseRowItem)->addHtml((string) $tmp));
			}
		} catch (Exception $e) {
		}

		return $formHead;
	}

	/**
	 * Renders validation errors (per form or per control).
	 *
	 * @param IControl|BaseControl|null $control
	 * @param bool                      $own
	 *
	 * @return string
	 */
	public function renderErrors($control = null, $own = true)
	{
		$errors = $control
			? $control->getErrors()
			: ($own ? $this->form->getOwnErrors() : $this->form->getErrors());
		if (!$errors) {
			return "";
		}
		$container = $this->getWrapper($control ? 'control errorcontainer' : 'error container');
		$item      = $this->getWrapper($control ? 'control erroritem' : 'error item');

		foreach ($errors as $error) {
			$item = clone $item;
			//            $closeBtn = $this->getWrapper('error dismiss');
			//            $closeIn = $this->getWrapper('error dismiss_in')
			//                            ->setHtml($this->getValue('error dismiss_content'));
			//            $closeBtn->setHtml($closeIn);
			//            $item->addHtml($closeBtn);

			if ($error instanceof Html) {
				$item->addHtml($error);
			} else {
				$item->addText($error);
			}
			$container->addHtml($item);
		}

		return "\n" . $container->render($control ? 1 : 0);
	}

	/**
	 * Renders form body.
	 * @return string
	 */
	public function renderBody()
	{
		$s = $remains = '';

		$defaultContainer = $this->getWrapper('group container');
		$translator       = $this->form->getTranslator();

		foreach ($this->form->getGroups() as $group) {
			if (!$group->getControls() || !$group->getOption('visual')) {
				continue;
			}

			$container = $group->getOption('container', $defaultContainer);
			$container = $container instanceof Html ? clone $container : Html::el($container);

			$id = $group->getOption('id');
			if ($id) {
				$container->id = $id;
			}

			$s .= "\n" . $container->startTag();

			$text = $group->getOption('label');
			if ($text instanceof Html) {
				$s .= $this->getWrapper('group label')->addHtml($text);
			} else if (is_string($text)) {
				if ($translator !== null) {
					$text = $translator->translate($text);
				}
				$s .= "\n" . $this->getWrapper('group label')->setText($text) . "\n";
			}

			$text = $group->getOption('description');
			if ($text instanceof Html) {
				$s .= $text;
			} else if (is_string($text)) {
				if ($translator !== null) {
					$text = $translator->translate($text);
				}
				$s .= $this->getWrapper('group description')->setText($text) . "\n";
			}

			$s .= $this->renderControls($group);

			$remains = $container->endTag() . "\n" . $remains;
			if (!$group->getOption('embedNext')) {
				$s       .= $remains;
				$remains = '';
			}
		}

		$s = $remains . $this->renderControls($this->form) . $s;

		$container = $this->getWrapper('form container');
		$container->setHtml($s);

		return $container->render(0);
	}

	public function renderControls(
		IControl|IComponent|ControlGroup|null $parent,
	): string
	{
		if (!($parent instanceof Container || $parent instanceof ControlGroup)) {
			throw new InvalidArgumentException(
				'Argument must be Nette\Forms\Container or Nette\Forms\ControlGroup instance.'
			);
		}

		$container = Html::el();
		$floating  = false;

		if ($parent instanceof BaseContainer) {
			$container = Html::el('fieldset');

			if ($parent->getLabel()) {
				$container->addHtml(
					Html::el(
						'legend',
						(string) $parent->getForm()
							->getTranslator()
							->translate($parent->getLabel()),
					),
				);
			}
		}

		if ($parent instanceof SaveCancelControl) {
			$container = $this->getWrapper('control save_cancel');
			$floating  = $parent->isFloating();
		}

		foreach ($parent->getControls() as $control) {
			if (
				$control->getOption('rendered')
				|| $control->getOption('type') === 'hidden'
				|| ($control->getForm(false) !== $this->form && $this->form !== null)
			) {
				// skip
			} else {
				$container->addHtml($this->renderPair($control));
			}
		}

		$s = '';
		if (count($container)) {
			$s .= "\n" . $container . "\n";
		}

		return $floating ? (string) Html::el('div', ['class' => 'floating-submit'])->setHtml($s) : $s;
	}

	/**
	 * @param BaseControl[] $controls
	 */
	public function renderPairMulti(array $controls): Html
	{
		$s    = [];
		$wrap = Html::el();
		foreach ($controls as $control) {
			if (is_string($control)) {
				$control = $this->form->getComponent($control, false);
			}

			if (!$control instanceof IControl) {
				throw new InvalidArgumentException('Argument must be array of Nette\Forms\IControl instances.');
			}
			/** @var BaseControl $control */
			$description = $control->getOption('description');
			if ($description instanceof Html) {
				$description = ' ' . $control->getOption('description');
			} else if (is_string($description)) {
				if ($control instanceof BaseControl) {
					$description = $control->translate($description);
				}
				$description = ' ' . $this->getWrapper('control description')->setText($description);
			} else {
				$description = '';
			}

			$control->setOption('rendered', true);
			$el = $control->getControl();
			if ($el instanceof Html && $el->getName() === 'input') {
				$el->class($this->getValue("control .$el->type"), true);
			}
			$s[] = $el . $description;

			$pair = $this->getWrapper('pair container');
			$pair->addHtml($this->renderLabel($control));
			$pair->addHtml($this->getWrapper('control container')->setHtml(implode(' ', $s)));

			$wrap->addHtml($pair);
		}

		return $wrap;
	}

	public function renderLabel(BaseControl $control): Html
	{
		if ($this->mode === RenderMode::SideBySideMode) {
			$label = $this->getWrapper('label side_container');
		} else if ($this->mode === RenderMode::VerticalMode) {
			$label = $this->getWrapper('label container');
		} else {
			$label = $this->getWrapper('label inline_container');
		}

		$controlLabel = $control->getLabel();
		if ($controlLabel instanceof Html && $this->mode === RenderMode::SideBySideMode) {
			$controlLabel->addClass($this->getValue('label side_class'));
		}
		$label->setHtml($controlLabel);

		if (empty($label->render()) && $this->renderMode === RenderMode::SideBySideMode) {
			$label = Html::el('div', [
				'class' => $this->getValue('label side_class'),
			]);
		}

		return $label;
	}

	public function renderPairs(array $controls): Html
	{
		$wrap = Html::el();

		foreach ($controls as $control) {
			if (is_string($control)) {
				$control = $this->form->getComponent($control, false);
			}

			if ($control instanceof BaseControl === false) {
				continue;
			}

			$wrap->addHtml($this->renderPair($control));
		}

		return $wrap;
	}

	public function renderPair(BaseControl $control): string
	{
		if ($control->getParent() && $control->getParent() instanceof SaveCancelControl) {
			$pair = Html::el();
		} else if ($this->mode == RenderMode::SideBySideMode) {
			$pair = $this->getWrapper('pair side_container');
		} else if ($this->mode == RenderMode::VerticalMode) {
			$pair = $this->getWrapper('pair container');
		} else {
			$pair = $this->getWrapper('pair inline_container');
		}

		$class = explode('\\', $control::class);
		$class = preg_replace('/Input$/', '', end($class));
		$pair->addClass('frm__type-' . Strings::webalize($class));

		if ($control instanceof BoolInput && $control->hasDefaultItems()) {
			$pair->addClass('frm__type-bool-default');
		}

		$pair->addHtml($this->renderLabel($control));

		if ($control instanceof CheckboxListInput) {
			$pair->addHtml(
				Html::el('div', ['data-simplebar' => true])
					->addHtml($this->renderControl($control)),
			);
		} else {
			$pair->addHtml($this->renderControl($control));
		}
		$pair->addHtml($this->renderDescription($control));
		$pair->class($control->hasErrors() ? $this->getValue('pair .error') : null, true);
		$pair->class($control->getOption('class'), true);
		$pair->id = $control->getOption('id');

		if (is_array($pair->attrs['class'])) {
			$pair->attrs['class'] = implode(' ', array_keys($pair->attrs['class']));
		}

		if ($control->isRequired()) {
			$pair->attrs['class'] .= ' required';
		}

		$return = $pair->render(0);

		/** @var BaseForm $form */
		$form = $control->getForm();
		foreach ($form->getCustomData('renderAfterInput-' . $control->getName(), []) as $val) {
			if ($val instanceof BaseControl) {
				$return .= $this->renderPair($val);
			}
		}

		return $return;
	}

	public function renderControl(BaseControl $control): Html
	{
		if ($this->mode == RenderMode::SideBySideMode) {
			$body = $this->getWrapper('control side_container');
		} else {
			$body = $this->getWrapper('control container');
		}

		$description = $control->getOption('description');
		if ($description instanceof Html) {
			$description = ' ' . $description;
		} else if (is_string($description)) {
			$description = $control->translate($description);
			$description = ' ' . $this->getWrapper('control description')->setText($description);
		} else {
			$description = '';
		}

		$control->setOption('rendered', true);
		$el = $control->getControl();
		if ($el instanceof Html && in_array($el->getName(), ['input', 'textarea'])) {
			$el->setAttribute('autocomplete', $this->autocomplete ? "on" : "off");
		}

		if (empty($body->render()) && $this->mode === RenderMode::VerticalMode) {
			// some would be inline otherwise
			if ($control instanceof UploadInput) {
				$body = $this->getWrapper('control .file');
			} else if ($control instanceof SelectInput) {
				$body = $this->getWrapper('control .select');
			}
		}

		return $body->setHtml($el . $description . $this->renderErrors($control));
	}

	/**
	 * @param IControl|BaseControl $control
	 */
	public function renderDescription($control): Html
	{
		if (!isset($control->description) || !$control->description) {
			return Html::el();
		}

		$wrapper = $this->getWrapper('control description');

		if (method_exists($control, 'getDescription')) {
			$wrapper->addHtml($control->getDescription());
		}

		return $wrapper;
	}

	/**
	 * Renders form end.
	 * @return string
	 */
	public function renderEnd()
	{
		$s = '';
		foreach ($this->form->getControls() as $control) {
			if ($control->getOption('type') === 'hidden' && !$control->getOption('rendered')) {
				$s .= $control->getControl();
			}
		}
		if (iterator_count($this->form->getComponents(true, TextInput::class)) < 2) {
			$s .= '<!--[if IE]><input type=IEbug disabled style="display:none"><![endif]-->';
		}
		if ($s) {
			$s = $this->getWrapper('hidden container')->setHtml($s) . "\n";
		}

		return $s . $this->form->getElementPrototype()->endTag() . "\n";
	}

	public function setColumns(int $label, ?int $control = null): void
	{
		if ($control === null) {
			$control = 12 - $label;
		}

		$this->labelColumns   = $label;
		$this->controlColumns = $control;

		$this->wrappers['control']['side_container'] = "div class=col-sm-$control";
		$this->wrappers['label']['side_container']   = "div class=col-sm-$label";
	}

	public function addToBaseExtendedLayout(string $column, string $name): void
	{
		$this->extendedLayout[self::BASE_EXTENDED_NAME][$column][] = $name;
	}
}
