<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Model\Lang\Lang;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Traits\TInputExtend;
use Nette\Forms\Controls\Checkbox;
use Nette\Utils\Html;

/**
 * @method BaseForm getForm(bool $throw = true)
 */
class CheckboxInput extends Checkbox
{
	use TInputExtend;

	/**
	 * @return Html
	 */
	public static function makeCheckbox(
		string  $name,
		        $caption,
		bool    $checked = false,
		        $value = false,
		bool    $required = false,
		bool    $disabled = false,
		string  $htmlId = '',
		?string $netteRules = null,
	)
	{
		$wrap  = Html::el('div', ['class' => 'pretty p-check']);
		$label = Html::el('label', ['for' => $htmlId])
			->setText($caption);
		$input = Html::el('input', [
			'type'     => 'checkbox',
			'class'    => 'form-check-input',
			'name'     => $name,
			'disabled' => $disabled,
			'required' => $required,
			'checked'  => $checked,
			'id'       => $htmlId,
		]);

		if ($netteRules) {
			$input->setAttribute('data-nette-rules', $netteRules);
		}

		if ($value !== false) {
			$input->attrs += [
				'value' => $value,
			];
		}

		$wrap->addHtml($input)
			->addHtml(
				Html::el('div class=state')
					->addHtml(Html::el('i class="icon fas fa-check"'))
					->addHtml($label),
			);

		return $wrap;
	}

	public function getControl(): Html
	{
		parent::getControl();

		$caption = $this->translate($this->caption);

		if (!$this->getIsMultiLanguage()) {
			return self::makeCheckbox(
				$this->getHtmlName(),
				$caption,
				$this->value,
				false,
				$this->required,
				$this->disabled,
				(string) $this->getHtmlId(),
				$this->getAttrRulesJson(),
			);
		} else {
			$wrap        = Html::el('div');
			$langs       = $this->getForm()->langsService->getLangs();
			/** @var Lang $defaultLang */
			$defaultLang = array_shift($langs);

			$selectedLang = $_COOKIE['contentLang'] ?? $defaultLang->getTag();
			if (!array_key_exists($selectedLang, $langs)) {
				$selectedLang = $defaultLang;
			}

			foreach ($langs as $k => $lang) {
				$getValue = $this->getValue();
				$value    = (is_array($getValue) ? $getValue[$k] : $getValue->$k) ?? '';

				$c = self::makeCheckbox(
					$this->getHtmlName() . "[$k]",
					$caption,
					!empty($value),
					false,
					$this->required,
					$this->disabled,
					$this->getHtmlId() . "-$k",
					$this->getAttrRulesJson(),
				);

				if ($defaultLang->getTag() !== $lang->getTag()) {
					unset($c->attrs['required']);

					foreach ($c->attrs['data-nette-rules'] as $k2 => $rule) {
						if ($rule['op'] == ':filled') {
							unset($c->attrs['data-nette-rules'][$k2]);
						}

						break;
					}
				}

				$c->attrs['data-content-lang'] = $k;

				if ($k !== $selectedLang) {
					$c->attrs['style'] = 'display: none;';
				}

				$wrap->addHtml($c);
			}

			return $wrap;
		}
	}

	/**
	 * @param mixed $type
	 */
	protected function getHttpData($type, ?string $htmlTail = null): mixed
	{
		if ($this->getIsMultiLanguage()) {
			$return = array_map(fn($lang) => $this->getForm()
				->getHttpData($type, $this->getHtmlName() . "[{$lang->getTag()}]" . $htmlTail),
				$this->getForm()->langsService->getLangs(false));

			return array_map(static fn($val) => $val !== null ? '1' : null, $return);
		} else {
			return parent::getHttpData($type, $htmlTail);
		}
	}

	public function setValue($value): self
	{
		if (!$this->getIsMultiLanguage()) {
			if (is_array($value)) {
				$value = array_values($value)[0];
			}

			return parent::setValue($value);
		}

		$this->value = $value;

		return $this;
	}

}
