<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use DateTimeImmutable;
use Core\Model\UI\Form\Enums\DateTimeFormat;
use DateTime;
use Nette\Forms\Controls\DateTimeControl;

class DateTimeInput extends DateTimeControl
{
	final public const DEFAULT_FORMAT = DateTimeFormat::D_DMY_DOTS_NO_LEAD . ' ' . DateTimeFormat::T_24_NO_LEAD;

	protected string $format      = 'd.m.yyyy h:mm';
	protected bool   $isValidated = false;

	public function __construct($label, int $type = self::TypeDate, bool $withSeconds = false)
	{
		parent::__construct($label, $type, $withSeconds);

		$this->addRule(fn($input,
		) => DateTimeFormat::validate($this->format, $input->value), 'invalid/incorrect format');

		$this->setFormat(self::DEFAULT_FORMAT);
	}

	public function cleanErrors(): void
	{
		$this->isValidated = false;
	}

	public function getValue(): DateTimeImmutable|string|int|null
	{
		$val = parent::getValue();
		if (!$this->isValidated) {
			return $val;
		}

		$value = null;
		if ($val instanceof DateTimeImmutable) {
			$value = DateTime::createFromImmutable($val);
		} else if ($val) {
			$value = DateTime::createFromFormat($this->format, (string) $val);
		}

		if ($value === false) {
			return null;
		}

		/** @phpstan-ignore-next-line  */
		return $value;
	}

	/**
	 * @param DateTime|null $value
	 */
	public function setValue($value): static
	{
		if ($value instanceof DateTime) {
			parent::setValue($value->format($this->format));

			return $this;
		} else if (is_string($value) && DateTimeFormat::validate($this->format, $value)) {
			parent::setValue($value);

			return $this;
		} else if ($value === null) {
			parent::setValue(null);

			return $this;
		} else {
			// this will fail validation test, but we don't want to throw an exception here
			parent::setValue($value);

			return $this;
		}
	}

	public function validate(): void
	{
		parent::validate();
		$this->isValidated = true;
	}

	/**
	 * @return string
	 */
	public function getFormat()
	{
		return $this->format;
	}

	public function setFormat(string $format, ?string $placeholder = null): static
	{
		$this->format = $format;

		if (method_exists($this, 'setPlaceholder')) {
			if ($placeholder === null) {
				$placeholder = self::makeFormatPlaceholder($format);
			}
			if (!empty($placeholder)) {
				$this->setPlaceholder($placeholder);
			}
		}

		return $this;
	}

	/**
	 * Turns datetime format into a placeholder,  e.g. 'd.m.Y' => 'dd.mm.yyyy'.
	 * Supported values: d, j, m, n, Y, y, a, A, g, G, h, H, i, s, c, U
	 *
	 * @param string $format
	 * @param bool   $example       whether to use e.g. 'd' or '01'
	 * @param bool   $appendExample attach example at the end of placeholder (true) or replace (false)e?
	 *
	 * @return string
	 */
	public static function makeFormatPlaceholder($format, $example = true, $appendExample = true)
	{
		$letterSubs = [
			'd' => 'dd',
			'j' => 'd',
			'm' => 'mm',
			'n' => 'm',
			'Y' => 'yyyy',
			'y' => 'yy',
			'a' => 'am/pm',
			'A' => 'AM/PM',
			'g' => 'h',
			'G' => 'h',
			'h' => 'hh',
			'H' => 'hh',
			'i' => 'mm',
			's' => 'ss',
			'c' => 'yyyy-mm-ddThh:mm:ss+hh:mm',
			'U' => 'unix timestamp',
		];
		$numSubs    = [
			'd' => '31',
			'j' => '31',
			'm' => '12',
			'n' => '12',
			'Y' => '1998',
			'y' => '98',
			'a' => 'am',
			'A' => 'AM',
			'g' => '12',
			'G' => '23',
			'h' => '12',
			'H' => '23',
			'i' => '59',
			's' => '00',
			'c' => '2012-12-21T17:42:00+00:00',
			'U' => '1501444136',
		];
		$letters    = strtr($format, $letterSubs);
		$ex         = strtr($format, $numSubs);

		if (!$example) {
			return $letters;
		} else if ($appendExample) {
			return "$letters ($ex)";
		}

		return $ex;
	}
}
