<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Traits;

use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\ButtonInput;
use Core\Model\UI\Form\Controls\CheckboxInput;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\DateTimeInput;
use Core\Model\UI\Form\Controls\MultiselectInput;
use Core\Model\UI\Form\Controls\PhoneInput;
use Core\Model\UI\Form\Controls\RadioInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Core\Model\UI\Form\Controls\SubmitButtonInput;
use Core\Model\UI\Form\Controls\TextAreaInput;
use Core\Model\UI\Form\Controls\TextInput;
use Core\Model\UI\Form\Controls\UploadInput;
use Nette\ComponentModel\IComponent;
use Nette\Forms\Container;
use Nette\Forms\Controls\BaseControl;
use Nette\Forms\Controls\Button;
use Nette\Forms\Controls\Checkbox;
use Nette\Forms\Controls\CheckboxList;
use Nette\Forms\Controls\MultiSelectBox;
use Nette\Forms\Controls\RadioList;
use Nette\Forms\Controls\SelectBox;
use Nette\Forms\Controls\SubmitButton;
use Nette\Forms\Controls\TextArea;
use Nette\Forms\Controls\TextInput as NTextInput;
use Nette\Forms\Controls\UploadControl;
use Nette\Forms\Form;
use Stringable;

trait BootstrapContainerTrait
{
	protected ?string $label = null;

	/**
	 * @return string
	 */
	public function getLabel() { return $this->label; }

	/**
	 * @param string $label
	 *
	 * @return $this
	 */
	public function setLabel($label)
	{
		$this->label = $label;

		return $this;
	}

	public function addButton(string $name, $caption = null): Button
	{
		$comp = new ButtonInput($caption);
		$comp->setBtnClass('btn-secondary');
		$this->addComponent($comp, $name);

		return $comp;
	}

	abstract public function addComponent(IComponent $component, ?string $name, ?string $insertBefore = null);

	public function addCheckbox(string $name, mixed $caption = null): Checkbox
	{
		$comp = new CheckboxInput($caption);
		$this->addComponent($comp, $name);

		return $comp;
	}

	public function addCheckboxList(string $name, mixed $label = null, array $items = null): CheckboxList
	{
		$comp = new CheckboxListInput($label, $items);
		$this->addComponent($comp, $name);

		return $comp;
	}

	/**
	 *
	 * @return BaseContainer
	 */
	public function addContainer($name): Container
	{
		$control               = new BaseContainer();
		$control->currentGroup = $this->currentGroup;
		if ($this->currentGroup !== null) {
			$this->currentGroup->add($control);
		}

		return $this[(string) $name] = $control;
	}

	/**
	 *
	 * @return BaseContainer
	 */
	public function addContainerWithLabel(string|int $name, ?string $label = null): Container
	{
		$control               = new BaseContainer();
		$control->currentGroup = $this->currentGroup;
		if ($this->currentGroup !== null) {
			$this->currentGroup->add($control);
		}
		$control->setLabel($label);

		return $this[(string) $name] = $control;
	}

	/**
	 * Adds a datetime input.
	 *
	 * @param string $name  name
	 * @param string $label label
	 */
	public function addDateTime(string $name, $label = null, bool $withSeconds = false): DateTimeInput
	{
		$comp = new DateTimeInput($label, DateTimeInput::TypeDateTime, $withSeconds);
		$this->addComponent($comp, $name);

		return $comp;
	}

	public function addEmail(
		string                 $name,
		string|Stringable|null $label = null,
		int                    $maxLength = 255,
	): NTextInput
	{
		return $this->addText($name, $label)
			->addRule(Form::EMAIL);
	}

	public function addText(string $name, string|Stringable|null $label = null, ?int $cols = null, ?int $maxLength = null): TextInput
	{
		$comp = new TextInput($label);
		$this->addComponent($comp, $name);

		return $comp;
	}

	public function addInputError(string $componentName, string $message): void
	{
		if ($this[$componentName] instanceof BaseControl) {
			$this[$componentName]->addError($message);
		}
	}

	public function addInteger($name, $label = null): TextInput
	{
		$control = $this->addText($name, $label);
		$control->addCondition(Form::FILLED)
			->addRule(Form::INTEGER);

		return $control;
	}

	public function addMultiSelect($name, $label = null, array $items = null, $size = null): MultiselectInput
	{
		$comp = new MultiselectInput($label, $items);
		$this->addComponent($comp, $name);

		return $comp;
	}

	/**
	 * @param string $name
	 * @param string $label
	 *
	 * @return UploadInput
	 */
	public function addMultiUpload($name, $label = null): UploadControl
	{
		return $this->addUpload($name, $label, true);
	}

	/**
	 * @param string $name
	 * @param string $label
	 * @param bool   $multiple
	 *
	 * @return UploadInput
	 */
	public function addUpload($name, $label = null, $multiple = false): UploadControl
	{
		$comp = new UploadInput($label, $multiple);
		$this->addComponent($comp, $name);

		return $comp;
	}

	/** phone number input with validation
	 *
	 * @param string $name
	 * @param string $label
	 */
	public function addPhone($name, $label): PhoneInput
	{
		$comp = new PhoneInput($label);
		$this->addComponent($comp, $name);

		return $comp;
	}

	public function addPassword(string $name, $label = null, ?int $cols = null, ?int $maxLength = null): NTextInput
	{
		return $this->addText($name, $label)
			->setHtmlAttribute('type', 'password');
	}

	public function addRadioList(string $name, $label = null, ?array $items = null): RadioList
	{
		$comp = new RadioInput($label, $items);
		$this->addComponent($comp, $name);

		return $comp;
	}

	/**
	 * @param string $label
	 * @param null   $size ignore
	 *
	 * @return SelectInput
	 */
	public function addSelect(string $name, $label = null, array $items = null, $size = null): SelectBox
	{
		$comp = new SelectInput($label, $items);
		$this->addComponent($comp, $name);

		return $comp;
	}

	/**
	 * @param string $name
	 * @param string $caption
	 * @param string $btnClass secondary button class (primary is 'btn')
	 *
	 * @return SubmitButtonInput
	 */
	public function addSubmit($name, $caption = null, $btnClass = 'btn-primary'): SubmitButton
	{
		$comp = new SubmitButtonInput($caption);
		$comp->setBtnClass($btnClass);
		$this->addComponent($comp, $name);

		return $comp;
	}

	/**
	 * @param string $name
	 * @param string $caption
	 * @param string $btnClass secondary button class (secondary is 'btn')
	 */
	public function addCancel($name, $caption = null, $btnClass = 'btn-secondary'): SubmitButtonInput
	{
		$comp = new SubmitButtonInput($caption);
		$comp->setBtnClass($btnClass);
		$this->addComponent($comp, $name);
		$comp->setValidationScope(null);
		$comp->onClick[] = function() use ($comp) {
			$form = $comp->getForm();
			if ($form instanceof BaseForm) {
				$form->onCancel($form);
			}
		};

		return $comp;
	}

	/**
	 * @return TextAreaInput
	 */
	public function addTextArea(
		string                 $name,
		string|Stringable|null $label = null,
		?int                   $cols = null,
		?int                   $rows = null,
	): TextArea
	{
		$comp = new TextAreaInput($label);
		$this->addComponent($comp, $name);

		return $comp;
	}
}
