<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Traits;

use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\CheckboxListNestedInput;
use Core\Model\UI\Form\Controls\SortableCheckboxList;
use Nette\Forms\Form;
use Nette\InvalidArgumentException;
use Nette\Utils\Html;

trait ChoiceInputTraitBase
{
	protected array $rawItems = [];

	/** @var null|array|string */
	protected $rawValue = null;

	public function flatAssocArray(array $array): array
	{
		$ret = [];
		foreach ($array as $key => $value) {
			if (is_array($value)) {
				if (isset($value['title'])) {
					$ret[$key] = $value['title'];
				} else {
					$ret += $this->flatAssocArray($value);
				}
			} else {
				$ret[$key] = $value;
			}
		}

		return $ret;
	}

	/**
	 * Makes array of &lt;option&gt;. Can handle associative arrays just fine. Checks for duplicate values.
	 *
	 * @param array    $items
	 * @param callable $optionArgs     takes ($value,$caption) and spits out an array of &lt;option&gt;
	 *                                 attributes
	 * @param array    $valuesRendered for internal use. Do not change.
	 *
	 * @return array
	 * @throws InvalidArgumentException when $items have multiple of the same values
	 */
	public function makeOptionList($items, callable $optionArgs, array &$valuesRendered = [])
	{
		$ret = [];
		foreach ($items as $value => $caption) {
			if (is_int($value)) {
				$value = (string) $value;
			}

			if (is_array($caption)) {
				// subgroup
				$option = Html::el('optgroup', ['label' => $value]);

				// options within the group
				$nested = $this->makeOptionList($caption, $optionArgs, $valuesRendered);

				foreach ($nested as $item) {
					$option->addHtml($item);
				}
			} else {
				$caption = $this->translate($caption);
				if (in_array($value, $valuesRendered)) {
					throw new InvalidArgumentException("Value '$value' is used multiple times.");
				}
				$valuesRendered[] = $value;

				// normal option
				$option = Html::el(
					'option',
					array_merge(['value' => (string) $value], $optionArgs($value, $caption)),
				);
				$option->setText($caption);
			}
			$ret[] = $option;
		}

		return $ret;
	}

	/**
	 * @return static
	 */
	public function setItems(array $items, bool $useKeys = true)
	{
		$this->rawItems = $items;

		$processed = $this->flatAssocArray($items);
		/** @noinspection PhpUndefinedMethodInspection */
		parent::setItems($processed, $useKeys);

		return $this;
	}

	protected function isControlDisabled(): bool
	{
		if (is_bool($this->disabled)) {
			return $this->disabled;
		}

		return false;
	}

	protected function isValueDisabled(string|int $value): bool
	{
		$disabled = $this->disabled;
		if (is_array($disabled)) {
			return isset($disabled[$value]) && $disabled[$value];
		} else if (!is_bool($disabled)) {
			/** @phpstan-ignore-next-line */
			return $disabled == $value;
		}

		return false;
	}

	protected function isValueSelected(int|string|null $value, ?string $lang = null): bool
	{
		if ($value === null) {
			return false;
		}

		$val = $this->getValue();

		if (is_array($val)) {
			if ($lang) {
				return $val[$lang] == $value;
			}

			return in_array($value, $val, false);
		}

		/** @phpstan-ignore-next-line */
		return $value == $val;
	}

	/**
	 * Sets control's value.
	 * @return static
	 * @internal
	 */
	public function setValue($value)
	{
		$this->value = $value;
		/** @phpstan-ignore-next-line */
		$this->rawValue = $this->getIsMultiLanguage() ? (array) $value : (string) $value;

		return $this;
	}

	public function loadHttpData(): void
	{
		$this->value = $this->getHttpData(Form::DATA_TEXT);

		$allowArray = in_array(static::class, [CheckboxListInput::class, SortableCheckboxList::class,
			CheckboxListNestedInput::class]);

		if ($this->value !== null && !$allowArray) {
			if (is_array($this->disabled) && isset($this->disabled[$this->value])) {
				$this->value = null;
			} else if (!$this->getIsMultiLanguage() || !is_array($this->value)) {
				$this->value = key([$this->value => null]);
			}
		}
	}
}
