<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Cookies;

use Core\AdminModule\Model\Sites;
use Core\FrontModule\Model\CookieServices;
use Core\Model\CookieAdditionalServices;
use Core\Model\Entities\CookieAdditionalService;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Tracy\Debugger;

class ServiceForm extends BaseControl
{
	protected CookieAdditionalServices $cookieAdditionalServices;
	protected CookieServices           $frontCookieServices;
	protected Sites                    $adminSites;
	protected ?CookieAdditionalService $service = null;

	public function __construct(
		CookieAdditionalServices $cookieAdditionalServices,
		CookieServices           $frontCookieServices,
		Sites                    $adminSites
	)
	{
		$this->cookieAdditionalServices = $cookieAdditionalServices;
		$this->frontCookieServices      = $frontCookieServices;
		$this->adminSites               = $adminSites;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addSelect('ident', 'core.serviceForm.name', $this->getIdentPairs())
			->setPrompt('default.choose')
			->setRequired();

		$sites = $this->adminSites->getOptionsForSelect();
		if (count($sites) > 1) {
			$form->addCheckboxList('sites', 'core.serviceForm.site', $sites)
				->setRequired();
		}

		$form->addHidden('id');

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, array $values): void
	{
		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$services = $this->getIdentPairs();

		if (!array_key_exists($values->ident, $services)) {
			return;
		}
		$serviceName = $services[$values->ident];
		if (Validators::isNone($values->id)) {
			$service = new CookieAdditionalService($values->ident, $serviceName);
			foreach ($values->sites ?? [] as $siteIdent) {
				$service->addSite($this->adminSites->get($siteIdent));
			}
		} else {
			$service = $this->cookieAdditionalServices->get((int) $values->id);

			$insertDiff = array_diff($values->sites, $service->getSitesIdent());
			$deleteDiff = array_diff($service->getSitesIdent(), $values->sites);
			foreach ($insertDiff as $siteIdent) {
				if ($site = $this->adminSites->get($siteIdent)) {
					$service->addSite($site);
					$this->em->persist($site);
				}
			}
			foreach ($deleteDiff as $siteIdent) {
				if ($site = $this->adminSites->get($siteIdent)) {
					$this->em->persist($site);
				}
			}
		}
		$service->name = $serviceName;

		$this->em->persist($service)->flush();

		$this->frontCookieServices->getCache()->clean([Cache::TAGS => ['cookieAdditionalServices']]);
	}

	public function setService(CookieAdditionalService $service): void
	{
		$form = $this['form'];

		$this->service = $service;

		if (isset($form['sites'])) {
			$form->getComponent('ident')->setItems($this->getIdentPairs());
		}

		$form->setDefaults([
			'id'    => $service->getId(),
			'ident' => $service->ident,
			'sites' => $service->getSitesIdent(),
		]);
	}

	protected function getIdentPairs(): array
	{
		$defaults = $this->cookieAdditionalServices->getAllWithDefaults();
		$result   = [];
		try {
			$json = file_get_contents('https://api-evidence.pshk.cz/v1/cms/cookies/services');
			if (!$json) {
				throw new \Exception('Failed load cookies services from api');
			}

			$data = Json::decode($json, Json::FORCE_ARRAY);

			if ($data['status'] !== 'success') {
				return $result;
			}

			foreach ($data['data'] as $k => $v) {
				if (!array_key_exists($k, $defaults) || ($this->service && $this->service->ident === $k)) {
					$result[$k] = $v['name'];
				}
			}
		} catch (\Exception $exception) {
			Debugger::log($exception);
		}

		return $result;
	}

}
