<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Variables;

use Core\AdminModule\Model\Variables;
use Core\Model\Entities\Variable;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Utils\ArrayHash;
use Core\Model\Templating\Variables as TemplatingVariables;

class VariableForm extends BaseControl
{
	/** @var string|null @persistent */
	public ?string $id = null;

	protected Variables $variables;
	protected Storage   $storage;

	public function __construct(
		Variables $variables,
		Storage   $storage
	)
	{
		$this->variables = $variables;
		$this->storage   = $storage;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('key', 'core.variable.key')
			->setDescription('core.variable.keyDescription')
			->setRequired();
		$form->addText('value', 'core.variable.value')
			->setDescription('core.variable.valueDescription')
			->setIsMultilanguage();

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, array $values): void
	{
		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$this->em->beginTransaction();
		try {
			$key = (string) $values->key;
			$key = Strings::toAscii($key);
			$key = Strings::toCamelCase($key);

			if ($this->id) {
				$existLang = [];
				foreach ($this->variables->getEr()->createQueryBuilder('v')
					         ->where('v.key = :key')
					         ->setParameter('key', $this->id)
					         ->getQuery()->getResult() as $row) {
					/** @var Variable $row */
					$row->key = $key;
					$row->setValue((string) $values->value[$row->lang]);

					$this->em->persist($row);
					$existLang[] = $row->lang;
				}

				foreach ($values->value as $lang => $value) {
					if (Arrays::contains($existLang, $lang)) {
						continue;
					}

					$variable = new Variable($key, $lang, (string) $value);
					$this->em->persist($variable);
				}
			} else {
				foreach ($values->value as $lang => $value) {
					$variable = new Variable($key, $lang, (string) $value);
					$this->em->persist($variable);
				}
			}

			$this->em->flush();
			$this->em->commit();
			$form->addCustomData('key', $key);

			$cache = new Cache($this->storage, TemplatingVariables::cacheNamespace);
			$cache->remove(TemplatingVariables::cacheNamespace);
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	public function setVariable(string $id): void
	{
		$this->id = $id;
		$defaults = [
			'key' => $id,
		];

		foreach ($this->variables->getEr()->createQueryBuilder('v')
			         ->where('v.key = :key')
			         ->setParameter('key', $this->id)
			         ->getQuery()->getArrayResult() as $row) {
			$defaults['value'][$row['lang']] = $row['value'];
		}

		$this['form']->setDefaults($defaults);
	}
}
